// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */
#include <linux/module.h>
#include <linux/pci.h>
#include <linux/debugfs.h>
#include <linux/dcache.h>
#include <linux/seq_file.h>
#include <linux/slab.h>
#include "net_driver.h"
#include "debugfs.h"
#include "nic.h"
#include "efct_auxbus.h"

/* Parameter definition bound to a structure - each file has one of these */
struct efct_debugfs_bound_param {
	const struct efct_debugfs_parameter *param;
	void *(*get_struct)(void *ref, u32 index);
	void *ref;
	u32 index;
};

/* Maximum length for a name component or symlink target */
#define EFCT_DEBUGFS_NAME_LEN 32

/* Top-level debug directory ([/sys/kernel]/debug/xilinx_efct) */
static struct dentry *efct_debug_root;

/* Functions for printing various types of parameter. */

int efct_debugfs_read_int(struct seq_file *file, void *data)
{
	seq_printf(file, "%d\n", *(int *)data);
	return 0;
}

int efct_debugfs_read_s8(struct seq_file *file, void *data)
{
	seq_printf(file, "%d\n", *(s8 *)data);
	return 0;
}

int efct_debugfs_read_uint(struct seq_file *file, void *data)
{
	seq_printf(file, "%#x\n", *(u32 *)data);
	return 0;
}

int efct_debugfs_read_u16(struct seq_file *file, void *data)
{
	seq_printf(file, "%#x\n", *(u16 *)data);
	return 0;
}

int efct_debugfs_read_u32(struct seq_file *file, void *data)
{
	seq_printf(file, "%#x\n", *(u32 *)data);
	return 0;
}

int efct_debugfs_read_u64(struct seq_file *file, void *data)
{
	seq_printf(file, "%#llx\n", *(u64 *)data);
	return 0;
}

int efct_debugfs_read_u8(struct seq_file *file, void *data)
{
	seq_printf(file, "%#x\n", *(u8 *)data);
	return 0;
}

int efct_debugfs_read_uchar(struct seq_file *file, void *data)
{
	seq_printf(file, "%#x\n", *(unsigned char *)data);
	return 0;
}

int efct_debugfs_read_ulong(struct seq_file *file, void *data)
{
	seq_printf(file, "%#lx\n", *(unsigned long *)data);
	return 0;
}

int efct_debugfs_read_atomic(struct seq_file *file, void *data)
{
	u32 value = atomic_read((atomic_t *)data);

	seq_printf(file, "%#x\n", value);
	return 0;
}

int efct_debugfs_read_bool(struct seq_file *file, void *data)
{
	seq_printf(file, "%d\n", *(bool *)data);
	return 0;
}

int efct_debugfs_read_string(struct seq_file *file, void *data)
{
	seq_printf(file, "%s\n", (const char *)data);
	return 0;
}

int efct_debugfs_read_addr(struct seq_file *file, void *data)
{
	seq_printf(file, "%p\n", data);
	return 0;
}

static const char *const evq_type_names[] = {
	[EVQ_T_RX] =	"RX_EVENT_QUEUE",
	[EVQ_T_TX] =	"TX_EVENT_QUEUE",
	[EVQ_T_AUX] =	"AUX_EVENT_QUEUE",
	[EVQ_T_NONE] =	"NONE_EVENT_QUEUE",
};

static const u32 evq_type_max = sizeof(evq_type_names);

static int efct_debugfs_read_evq_type(struct seq_file *file, void *data)
{
	u32 value = *(enum evq_type *)data;

	seq_printf(file, "%d => %s\n", value,
		   STRING_TABLE_LOOKUP(value, evq_type));
	return 0;
}

#define EFCT_EVQ_TYPE_PARAMETER(container_type, parameter)		\
	EFCT_PARAMETER(container_type, parameter,			\
		      enum evq_type, efct_debugfs_read_evq_type)

static const char *const nic_state_names[] = {
	[STATE_UNINIT] =	"UNINIT",
	[STATE_PROBED] =	"PROBED",
	[STATE_NET_UP] =	"READY",
	[STATE_NET_DOWN] =	"DOWN",
	[STATE_DISABLED] =	"DISABLED",
	[STATE_NET_UP | STATE_RECOVERY] =	"READY_RECOVERY",
	[STATE_NET_DOWN | STATE_RECOVERY] =	"DOWN_RECOVERY",
	[STATE_PROBED | STATE_RECOVERY] =	"PROBED_RECOVERY",
	[STATE_NET_UP | STATE_FROZEN] =	"READY_FROZEN",
	[STATE_NET_DOWN | STATE_FROZEN] =	"DOWN_FROZEN",
};

static const u32 nic_state_max = sizeof(nic_state_names);

static int efct_debugfs_read_nic_state(struct seq_file *file, void *data)
{
	u32 value = *(enum nic_state *)data;

	seq_printf(file, "%d => %s\n", value,
		   STRING_TABLE_LOOKUP(value, nic_state));
	return 0;
}

#define EFCT_NIC_STATE_PARAMETER(container_type, parameter)		\
	EFCT_PARAMETER(container_type, parameter,			\
		      enum nic_state, efct_debugfs_read_nic_state)

static int efct_nic_debugfs_read_name(struct seq_file *file, void *data)
{
	struct efct_nic *efct = data;

	seq_printf(file, "%s\n", efct->name);
	return 0;
}

static const char *const driver_dist_layout_names[] = {
	[RX_LAYOUT_DISTRIBUTED] =	"RX_LAYOUT_DISTRIBUTED",
	[RX_LAYOUT_SEPARATED] =	"RX_LAYOUT_SEPERATED",
};

static const u32 driver_dist_layout_max = sizeof(driver_dist_layout_names);

static int efct_debugfs_read_dist_layout(struct seq_file *file, void *data)
{
	u32 value = *(enum driver_dist_layout *)data;

	seq_printf(file, "%d => %s\n", value,
		   STRING_TABLE_LOOKUP(value, driver_dist_layout));
	return 0;
}

#define EFCT_DIST_LAYOUT_PARAMETER(container_type, parameter)		\
	EFCT_PARAMETER(container_type, parameter,			\
		      enum driver_dist_layout, efct_debugfs_read_dist_layout)

#ifdef CONFIG_XILINX_AUX_EFCT
static int efct_debugfs_read_aux_client_name(struct seq_file *file, void *data)
{
	struct xlnx_efct_client *client = data;

	seq_printf(file, "%s\n", client->drvops->name);
	return 0;
}

static int efct_debugfs_read_aux_client_priv_data(struct seq_file *file, void *data)
{
	struct xlnx_efct_client *client = data;

	seq_printf(file, "%p\n", client->driver_priv);
	return 0;
}

static const char *const client_state_names[] = {
	[STATE_OPEN] =	"OPEN",
	[STATE_CLOSED] = "CLOSED",
};

static const u32 client_state_max = sizeof(client_state_names);

static int efct_debugfs_read_client_state(struct seq_file *file, void *data)
{
	u32 value = *(enum client_state *)data;

	seq_printf(file, "%d => %s\n", value,
		   STRING_TABLE_LOOKUP(value, client_state));
	return 0;
}

#define EFCT_CLIENT_STATE_PARAMETER(container_type, parameter)		\
	EFCT_PARAMETER(container_type, parameter,			\
		      enum client_state, efct_debugfs_read_client_state)

static int efct_debugfs_read_active_aux_clients(struct seq_file *file, void *data)
{
	struct efct_rx_queue *rxq = data;

	seq_printf(file, "%lx\n", rxq->active_aux_clients[0]);
	return 0;
}

static int efct_debugfs_read_aux_client_active_txq(struct seq_file *file, void *data)
{
	struct xlnx_efct_client *client = data;

	seq_printf(file, "%lx\n", client->active_txq[0]);
	return 0;
}

static int efct_debugfs_read_aux_client_active_evq(struct seq_file *file, void *data)
{
	struct xlnx_efct_client *client = data;

	seq_printf(file, "%lx\n", client->active_evq[0]);
	return 0;
}

static int efct_debugfs_read_aux_client_filter_map(struct seq_file *file, void *data)
{
	struct xlnx_efct_client *client = data;

	seq_printf(file, "%lx\n", client->filter_map[0]);
	return 0;
}

static int efct_debugfs_read_aux_client_index(struct seq_file *file, void *data)
{
	struct xlnx_efct_client *client;
	struct efct_super_buffer *sb;

	sb = data;
	client = sb->client_id;
	if (client)
		seq_printf(file, "%x\n", client->index);

	return 0;
}
#endif

/* Sequential file interface to bound parameters */
static int efct_debugfs_seq_show(struct seq_file *file, void *v)
{
	struct efct_debugfs_bound_param *binding = file->private;
	void *structure;
	int rc;

	rtnl_lock();
	structure = binding->get_struct(binding->ref, binding->index);
	if (structure)
		rc = binding->param->reader(file,
					    structure + binding->param->offset);
	else
		rc = -EINVAL;
	rtnl_unlock();
	return rc;
}

static int efct_debugfs_open(struct inode *inode, struct file *file)
{
	return single_open(file, efct_debugfs_seq_show, inode->i_private);
}

static const struct file_operations efct_debugfs_file_ops = {
	.owner   = THIS_MODULE,
	.open    = efct_debugfs_open,
	.read    = seq_read,
	.llseek  = seq_lseek,
	.release = single_release
};

static void *efct_debugfs_get_same(void *ref, u32 index)
{
	return ref;
}

/**
 * efct_fini_debugfs_child - remove a named child of a debugfs directory
 * @dir:		Directory
 * @name:		Name of child
 *
 * This removes the named child from the directory, if it exists.
 */
void efct_fini_debugfs_child(struct dentry *dir, const char *name)
{
	struct qstr child_name = QSTR_INIT(name, strlen(name));
	struct dentry *child;

	child = d_hash_and_lookup(dir, &child_name);
	if (!IS_ERR_OR_NULL(child)) {
		/* If it's a "regular" file, free its parameter binding */
		if (S_ISREG(child->d_inode->i_mode))
			kfree(child->d_inode->i_private);
		debugfs_remove(child);
		dput(child);
	}
}

/**
 * efct_init_debugfs_files - create parameter-files in a debugfs directory
 * @parent:		Containing directory
 * @params:		Pointer to zero-terminated parameter definition array
 * @ignore:		Bitmask of array entries to ignore
 * @get_struct:		Obtain the structure containing parameters
 * @ref:		Opaque reference passed into the @get_struct call
 * @struct_index:	Index passed into the @get_struct call
 *
 * Add parameter-files to the given debugfs directory.  Return a
 * negative error code or 0 on success.
 */
static int
efct_init_debugfs_files(struct dentry *parent,
			struct efct_debugfs_parameter *params, u64 ignore,
			void *(*get_struct)(void *ref, u32 index),
			void *ref, u32 struct_index)
{
	struct efct_debugfs_bound_param *binding;
	u32 pos;

	for (pos = 0; params[pos].name; pos++) {
		struct dentry *entry;

		if ((1ULL << pos) & ignore)
			continue;

		binding = kmalloc(sizeof(*binding), GFP_KERNEL);
		if (!binding)
			goto err;
		binding->param = &params[pos];
		binding->get_struct = get_struct;
		binding->ref = ref;
		binding->index = struct_index;

		entry = debugfs_create_file(params[pos].name, 0444, parent,
					    binding, &efct_debugfs_file_ops);
		if (!entry) {
			kfree(binding);
			goto err;
		}
	}

	return 0;

err:
	while (pos--) {
		if ((1ULL << pos) & ignore)
			continue;

		efct_fini_debugfs_child(parent, params[pos].name);
	}
	return -ENOMEM;
}

/* Per-card parameters */
static struct efct_debugfs_parameter efct_debugfs_design_parameters[] = {
	EFCT_UINT_PARAMETER(struct design_params, rx_stride),
	EFCT_UINT_PARAMETER(struct design_params, evq_stride),
	EFCT_UINT_PARAMETER(struct design_params, ctpio_stride),
	EFCT_UINT_PARAMETER(struct design_params, rx_buffer_len),
	EFCT_UINT_PARAMETER(struct design_params, rx_queues),
	EFCT_UINT_PARAMETER(struct design_params, tx_apertures),
	EFCT_UINT_PARAMETER(struct design_params, rx_buf_fifo_size),
	EFCT_UINT_PARAMETER(struct design_params, rx_metadata_len),
	EFCT_UINT_PARAMETER(struct design_params, tx_max_reorder),
	EFCT_UINT_PARAMETER(struct design_params, tx_aperture_size),
	EFCT_UINT_PARAMETER(struct design_params, unsol_credit_seq_mask),
	EFCT_UINT_PARAMETER(struct design_params, tx_fifo_size),
	EFCT_UINT_PARAMETER(struct design_params, frame_offset_fixed),
	EFCT_UINT_PARAMETER(struct design_params, ts_subnano_bit),
	EFCT_UINT_PARAMETER(struct design_params, l4_csum_proto),
	EFCT_UINT_PARAMETER(struct design_params, max_runt),
	EFCT_UINT_PARAMETER(struct design_params, evq_sizes),
	EFCT_UINT_PARAMETER(struct design_params, num_evq),
	{NULL},
};

static struct efct_debugfs_parameter efct_debugfs_card_parameters[] = {
	EFCT_UCHAR_PARAMETER(struct efct_device, num_ports),
	EFCT_UINT_PARAMETER(struct efct_device, mem_bar),
	EFCT_ADDR_PARAMETER(struct efct_device, membase),
	EFCT_U32_PARAMETER(struct efct_device, reg_base),
	EFCT_DIST_LAYOUT_PARAMETER(struct efct_device, dist_layout),
	EFCT_U8_PARAMETER(struct efct_device, separated_rx_cpu),
	EFCT_U16_PARAMETER(struct efct_device, vec_per_port),
	EFCT_BOOL_PARAMETER(struct efct_device, mcdi_logging),
	{NULL},
};

/* Per-NIC error counts */
static struct efct_debugfs_parameter efct_debugfs_nic_error_parameters[] = {
	EFCT_ATOMIC_PARAMETER(struct efct_nic_errors, missing_event),
	EFCT_ATOMIC_PARAMETER(struct efct_nic_errors, rx_reset),
	EFCT_ATOMIC_PARAMETER(struct efct_nic_errors, rx_desc_fetch),
	EFCT_ATOMIC_PARAMETER(struct efct_nic_errors, tx_desc_fetch),
	EFCT_ATOMIC_PARAMETER(struct efct_nic_errors, spurious_tx),
	{NULL},
};

static struct efct_debugfs_parameter efct_debugfs_link_state_parameters[] = {
	EFCT_BOOL_PARAMETER(struct efct_link_state, up),
	EFCT_BOOL_PARAMETER(struct efct_link_state, fd),
	EFCT_UINT_PARAMETER(struct efct_link_state, speed),
	EFCT_U32_PARAMETER(struct efct_link_state, ld_caps),
	EFCT_U32_PARAMETER(struct efct_link_state, lp_caps),
	{NULL},
};

/* Per-NIC port params */
static struct efct_debugfs_parameter efct_debugfs_port_parameters[] = {
	EFCT_UINT_PARAMETER(struct efct_nic, int_error_count),
	EFCT_ULONG_PARAMETER(struct efct_nic, int_error_expire),
	EFCT_U8_PARAMETER(struct efct_nic, port_num),
	EFCT_NIC_STATE_PARAMETER(struct efct_nic, state),
	EFCT_U8_PARAMETER(struct efct_nic, max_evq_count),
	EFCT_U8_PARAMETER(struct efct_nic, rxq_count),
	EFCT_U8_PARAMETER(struct efct_nic, max_txq_count),
	EFCT_UINT_PARAMETER(struct efct_nic, timer_max_ns),
	EFCT_UINT_PARAMETER(struct efct_nic, timer_quantum_ns),
	EFCT_U32_PARAMETER(struct efct_nic, msg_enable),
	EFCT_ADDR_PARAMETER(struct efct_nic, membase),
	EFCT_ADDR_PARAMETER(struct efct_nic, wc_membase),
	EFCT_U32_PARAMETER(struct efct_nic, port_base),
	EFCT_U16_PARAMETER(struct efct_nic, num_mac_stats),
	EFCT_BOOL_PARAMETER(struct efct_nic, stats_enabled),
	EFCT_U8_PARAMETER(struct efct_nic, wanted_fc),
	EFCT_UINT_PARAMETER(struct efct_nic, stats_period_ms),
	EFCT_UINT_PARAMETER(struct efct_nic, irq_mod_step_ns),
	EFCT_UINT_PARAMETER(struct efct_nic, irq_rx_moderation_ns),
	EFCT_BOOL_PARAMETER(struct efct_nic, stats_initialised),
	EFCT_UINT_PARAMETER(struct efct_nic, n_link_state_changes),
	{.name = "name",
	 .offset = 0,
	 .reader = efct_nic_debugfs_read_name},
	{NULL},
};

#ifdef CONFIG_XILINX_AUX_EFCT
/* Per-NIC port params */
static struct efct_debugfs_parameter efct_debugfs_aux_client_parameters[] = {
	{.name = "name",
	 .offset = 0,
	 .reader = efct_debugfs_read_aux_client_name},
	{.name = "driv_priv",
	 .offset = 0,
	 .reader = efct_debugfs_read_aux_client_priv_data},
	EFCT_CLIENT_STATE_PARAMETER(struct xlnx_efct_client, state),
	EFCT_UINT_PARAMETER(struct xlnx_efct_client, active_hugepages),
	EFCT_U8_PARAMETER(struct xlnx_efct_client, index),
	{.name = "active_txq",
	 .offset = 0,
	 .reader = efct_debugfs_read_aux_client_active_txq},
	{.name = "active_evq",
	 .offset = 0,
	 .reader = efct_debugfs_read_aux_client_active_evq},
	{.name = "filter_map",
	 .offset = 0,
	 .reader = efct_debugfs_read_aux_client_filter_map},
	{NULL},
};

static struct efct_debugfs_parameter efct_debugfs_client_rxq_info_parameters[] = {
	EFCT_ATOMIC_PARAMETER(struct efct_rxq_client_meta, active_hp),
	EFCT_UINT_PARAMETER(struct efct_rxq_client_meta, added_hp),
	EFCT_UINT_PARAMETER(struct efct_rxq_client_meta, removed_hp),
	EFCT_UINT_PARAMETER(struct efct_rxq_client_meta, ref_cnt),
	{NULL},
};

static struct efct_debugfs_parameter efct_debugfs_rx_queue_hpl_parameters[] = {
	EFCT_UINT_PARAMETER(struct efct_huge_page_list, max_hp_id),
	{NULL},
};

/* Aux device Parameters */
static struct efct_debugfs_parameter efct_debugfs_aux_dev_parameters[] = {
	EFCT_U16_PARAMETER(struct aux_resources, evq_base),
	EFCT_U16_PARAMETER(struct aux_resources, evq_lim),
	EFCT_U16_PARAMETER(struct aux_resources, txq_base),
	EFCT_U16_PARAMETER(struct aux_resources, txq_lim),
	{NULL},
};

/* SBL Parameters */
static struct efct_debugfs_parameter efct_debugfs_sbl_parameters[] = {
	EFCT_ADDR_PARAMETER(struct efct_super_buffer, dma_buffer.addr),
	EFCT_BOOL_PARAMETER(struct efct_super_buffer, dma_buffer.sentinel),
	EFCT_U32_PARAMETER(struct efct_super_buffer, dma_buffer.len),
	EFCT_U16_PARAMETER(struct efct_super_buffer, ref_count),
	EFCT_U16_PARAMETER(struct efct_super_buffer, flags),
	{.name = "index",
	 .offset = 0,
	 .reader = efct_debugfs_read_aux_client_index},
	{NULL},
};
#endif

/* NBL Parameters */
static struct efct_debugfs_parameter efct_debugfs_nbl_parameters[] = {
	EFCT_ADDR_PARAMETER(struct efct_nic_buffer, dma_buffer.addr),
	EFCT_BOOL_PARAMETER(struct efct_nic_buffer, dma_buffer.sentinel),
	EFCT_U32_PARAMETER(struct efct_nic_buffer, dma_buffer.len),
	EFCT_UCHAR_PARAMETER(struct efct_nic_buffer, is_dbl),
	EFCT_INT_PARAMETER(struct efct_nic_buffer, id),
	{NULL},
};

/* DBL Parameters */
static struct efct_debugfs_parameter efct_debugfs_dbl_parameters[] = {
	EFCT_ADDR_PARAMETER(struct efct_driver_buffer, dma_buffer.addr),
	EFCT_BOOL_PARAMETER(struct efct_driver_buffer, dma_buffer.sentinel),
	EFCT_U32_PARAMETER(struct efct_driver_buffer, dma_buffer.len),
	{NULL},
};

/* Rx Queue Parameters */
static struct efct_debugfs_parameter efct_debugfs_rx_queue_nbl_parameters[] = {
	EFCT_UINT_PARAMETER(struct efct_nic_buffer_list, active_nic_buffs),
	EFCT_UCHAR_PARAMETER(struct efct_nic_buffer_list, head_index),
	EFCT_UCHAR_PARAMETER(struct efct_nic_buffer_list, tail_index),
	EFCT_UINT_PARAMETER(struct efct_nic_buffer_list, meta_offset),
	EFCT_UINT_PARAMETER(struct efct_nic_buffer_list, prev_meta_offset),
	EFCT_UINT_PARAMETER(struct efct_nic_buffer_list, frame_offset_fixed),
	EFCT_UINT_PARAMETER(struct efct_nic_buffer_list, seq_no),
	{NULL},
};

static struct efct_debugfs_parameter efct_debugfs_rx_queue_parameters[] = {
	EFCT_UINT_PARAMETER(struct efct_rx_queue, evq_index),
	EFCT_UINT_PARAMETER(struct efct_rx_queue, cpu),
	EFCT_U8_PARAMETER(struct efct_rx_queue, label),
	EFCT_UINT_PARAMETER(struct efct_rx_queue, buffer_size),
	EFCT_UINT_PARAMETER(struct efct_rx_queue, filter_count),
	EFCT_U16_PARAMETER(struct efct_rx_queue, num_rx_buffs),
	EFCT_U32_PARAMETER(struct efct_rx_queue, num_entries),
	EFCT_U16_PARAMETER(struct efct_rx_queue, pkt_stride),
	EFCT_U64_PARAMETER(struct efct_rx_queue, n_rx_tcp_udp_chksum_err),
	EFCT_U64_PARAMETER(struct efct_rx_queue, n_rx_ip_hdr_chksum_err),
	EFCT_U64_PARAMETER(struct efct_rx_queue, n_rx_eth_crc_err),
	EFCT_U64_PARAMETER(struct efct_rx_queue, n_rx_sentinel_drop_count),
#ifdef CONFIG_XILINX_AUX_EFCT
	{.name = "active_aux_clients",
	 .offset = 0,
	 .reader = efct_debugfs_read_active_aux_clients},
	EFCT_U64_PARAMETER(struct efct_rx_queue, poison_cfg.value),
	EFCT_ULONG_PARAMETER(struct efct_rx_queue, poison_cfg.length),
	EFCT_S8_PARAMETER(struct efct_rx_queue, exclusive_client),
#endif
	{NULL},
};

/* Tx Queue Parameters */
static struct efct_debugfs_parameter efct_debugfs_tx_queue_parameters[] = {
	EFCT_UINT_PARAMETER(struct efct_tx_queue, evq_index),
	EFCT_U8_PARAMETER(struct efct_tx_queue, label),
	EFCT_UINT_PARAMETER(struct efct_tx_queue, piobuf_offset),
	EFCT_UCHAR_PARAMETER(struct efct_tx_queue, added_sequence),
	EFCT_UCHAR_PARAMETER(struct efct_tx_queue, completed_sequence),
	EFCT_ATOMIC_PARAMETER(struct efct_tx_queue, inuse_fifo_bytes),
	EFCT_ATOMIC_PARAMETER(struct efct_tx_queue, inflight_pkts),
	EFCT_U8_PARAMETER(struct efct_tx_queue, ct_thresh),
	{NULL},
};

/* Event Queue Parameters */
static struct efct_debugfs_parameter efct_debugfs_ev_queue_parameters[] = {
	EFCT_UINT_PARAMETER(struct efct_ev_queue, entries),
	EFCT_EVQ_TYPE_PARAMETER(struct efct_ev_queue, type),
	EFCT_U16_PARAMETER(struct efct_ev_queue, msi.irq),
	EFCT_U16_PARAMETER(struct efct_ev_queue, msi.idx),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, consumer_index),
	EFCT_UCHAR_PARAMETER(struct efct_ev_queue, queue_count),
	EFCT_BOOL_PARAMETER(struct efct_ev_queue, evq_phase),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, unsol_consumer_index),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, irq_moderation_ns),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, irq_count),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, irq_mod_score),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, irq_adapt_low_thresh),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, irq_adapt_high_thresh),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, irq_adapt_irqs),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, tx_merge_timeout_ns),
	EFCT_UINT_PARAMETER(struct efct_ev_queue, rx_merge_timeout_ns),
	{NULL},
};

/* efct_init_debugfs - create debugfs directories for x3 driver
 *
 * Create debugfs directory "xilinx_efct".  This must be
 * called before any of the other functions that create debugfs
 * directories.  Return a negative error code or 0 on success.  The
 * directories must be cleaned up using efct_fini_debugfs().
 */
int efct_init_debugfs(const char *module)
{
	int rc;

	/* Create top-level directory */
	efct_debug_root = debugfs_create_dir(module, NULL);
	if (!efct_debug_root) {
		pr_err("debugfs_create_dir %s failed.\n", module);
		rc = -ENOMEM;
		goto err;
	} else if (IS_ERR(efct_debug_root)) {
		rc = PTR_ERR(efct_debug_root);
		pr_err("debugfs_create_dir %s failed, rc=%d.\n", module, rc);
		goto err;
	}

	return 0;

 err:
	efct_fini_debugfs();
	return rc;
}

/**
 * efct_init_debugfs_func - create debugfs directory for PCIe function
 * @efct_dev:		efct NIC
 *
 * Create debugfs directory containing parameter-files for @efct_dev,
 * and a subdirectory "design_params" containing per-card design parameters.
 * Return a negative error code or 0 on success.  The directories
 * must be cleaned up using efct_fini_debugfs_nic().
 */
int efct_init_debugfs_func(struct efct_device *efct_dev)
{
	int rc;

	/* Create directory */
	efct_dev->debug_dir = debugfs_create_dir(pci_name(efct_dev->pci_dev),
						 efct_debug_root);
	if (IS_ERR(efct_dev->debug_dir)) {
		rc = PTR_ERR(efct_dev->debug_dir);
		efct_dev->debug_dir = NULL;
		goto err;
	}
	if (!efct_dev->debug_dir)
		goto err_mem;

	/* Create design_params directory */
	efct_dev->params.debug_dir = debugfs_create_dir("design_params",
							efct_dev->debug_dir);
	if (IS_ERR(efct_dev->params.debug_dir)) {
		rc = PTR_ERR(efct_dev->params.debug_dir);
		efct_dev->params.debug_dir = NULL;
		goto err;
	}
	if (!efct_dev->params.debug_dir)
		goto err_mem;

	/* Create files */
	rc = efct_init_debugfs_files(efct_dev->params.debug_dir,
				     efct_debugfs_design_parameters, 0,
				     efct_debugfs_get_same, efct_dev, 0);
	if (rc)
		goto err;

	rc = efct_init_debugfs_files(efct_dev->debug_dir,
				     efct_debugfs_card_parameters, 0,
				     efct_debugfs_get_same, efct_dev, 0);
	if (rc)
		goto err;

	return 0;

 err_mem:
	rc = -ENOMEM;
 err:
	efct_fini_debugfs_func(efct_dev);
	return rc;
}

/**
 * efct_init_debugfs_nic - create debugfs directory for NIC
 * @efct:		efct NIC
 *
 * Create "port" debugfs directory containing parameter-files for @efct,
 * a subdirectory "errors" containing per-NIC error counts and
 * a subdirectory "link state" containing per-NIC link state params
 * Return a negative error code or 0 on success.  The directories
 * must be cleaned up using efct_fini_debugfs_nic().
 */
int efct_init_debugfs_nic(struct efct_nic *efct)
{
	char name[EFCT_DEBUGFS_NAME_LEN];
	int rc;

	/* Create directory */
	if (snprintf(name, sizeof(name), "port%d", efct->port_num)
			>= sizeof(name))
		goto err_len;

	efct->debug_port_dir = debugfs_create_dir(name, efct->efct_dev->debug_dir);
	if (IS_ERR(efct->debug_port_dir)) {
		rc = PTR_ERR(efct->debug_port_dir);
		efct->debug_port_dir = NULL;
		goto err;
	}
	if (!efct->debug_port_dir)
		goto err_mem;

	/* Create errors directory */
	efct->errors.debug_dir = debugfs_create_dir("errors", efct->debug_port_dir);
	if (IS_ERR(efct->errors.debug_dir)) {
		rc = PTR_ERR(efct->errors.debug_dir);
		efct->errors.debug_dir = NULL;
		goto err;
	}
	if (!efct->errors.debug_dir)
		goto err_mem;

	efct->link_state.debug_dir = debugfs_create_dir("link_state", efct->debug_port_dir);
	if (IS_ERR(efct->link_state.debug_dir)) {
		rc = PTR_ERR(efct->link_state.debug_dir);
		efct->link_state.debug_dir = NULL;
		goto err;
	}
	if (!efct->link_state.debug_dir)
		goto err_mem;

    /* Create files */
	rc = efct_init_debugfs_files(efct->errors.debug_dir,
				     efct_debugfs_nic_error_parameters, 0,
				     efct_debugfs_get_same, &efct->errors, 0);
	if (rc)
		goto err;

	rc = efct_init_debugfs_files(efct->link_state.debug_dir,
				     efct_debugfs_link_state_parameters, 0,
				     efct_debugfs_get_same, &efct->link_state, 0);
	if (rc)
		goto err;

	rc = efct_init_debugfs_files(efct->debug_port_dir,
				     efct_debugfs_port_parameters, 0,
				     efct_debugfs_get_same, efct, 0);
	if (rc)
		goto err;

	return 0;

err_len:
	rc = -ENAMETOOLONG;
	goto err;
err_mem:
	rc = -ENOMEM;
err:
	efct_fini_debugfs_nic(efct);
	return rc;
}

#ifdef CONFIG_XILINX_AUX_EFCT
int efct_init_debugfs_aux_dev_client(struct xlnx_efct_client *client)
{
	char name[EFCT_DEBUGFS_NAME_LEN];
	int rc, i;

	/* Create directory */
	if (snprintf(name, sizeof(name), "client@%x", client->index)
			>= sizeof(name))
		goto err_len;

	client->debug_dir = debugfs_create_dir(name,
					       client->efct->debug_aux_dir);
	if (IS_ERR(client->debug_dir)) {
		rc = PTR_ERR(client->debug_dir);
		client->debug_dir = NULL;
		goto err;
	}
	if (!client->debug_dir)
		goto err_mem;

	rc = efct_init_debugfs_files(client->debug_dir,
				     efct_debugfs_aux_client_parameters, 0,
				     efct_debugfs_get_same, client, 0);
	if (rc)
		goto err;

	for (i = 0; i < client->efct->rxq_count; i++) {
		/* Create directory */
		if (snprintf(name, sizeof(name), "rxq_info%d", i)
				>= sizeof(name))
			goto err_len;
		client->rxq_info[i].debug_dir = debugfs_create_dir(name, client->debug_dir);
		if (IS_ERR(client->rxq_info[i].debug_dir)) {
			rc = PTR_ERR(client->rxq_info[i].debug_dir);
			client->rxq_info[i].debug_dir = NULL;
			goto err;
		}
		if (!client->rxq_info[i].debug_dir)
			goto err_mem;

		/* Create files */
		rc = efct_init_debugfs_files(client->rxq_info[i].debug_dir,
					     efct_debugfs_client_rxq_info_parameters,
					     0, efct_debugfs_get_same, &client->rxq_info[i], 0);
		if (rc)
			goto err;
	}

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efct_fini_debugfs_aux_dev_client(client);
	return rc;
}

/**
 * efct_init_debugfs_aux_dev - create debugfs directory for Aux Device
 * @efct:		efct NIC
 *
 * Create a aux device debugfs directory containing parameter-files
 * for @efct,
 * Return a negative error code or 0 on success.  The directories
 * must be cleaned up using efct_fini_debugfs_aux_dev().
 */
int efct_init_debugfs_aux_dev(struct efct_nic *efct)
{
	char name[EFCT_DEBUGFS_NAME_LEN];
	int rc;

	/* Create directory */
	if (snprintf(name, sizeof(name), "%s_%d", efct->aux_dev.adev.name,
		     efct->port_num)
			>= sizeof(name))
		goto err_len;

	efct->debug_aux_dir = debugfs_create_dir(name, efct->debug_port_dir);
	if (IS_ERR(efct->debug_aux_dir)) {
		rc = PTR_ERR(efct->debug_aux_dir);
		efct->debug_aux_dir = NULL;
		goto err;
	}
	if (!efct->debug_aux_dir)
		goto err_mem;

	/* Create files */
	rc = efct_init_debugfs_files(efct->debug_aux_dir,
				     efct_debugfs_aux_dev_parameters, 0,
				     efct_debugfs_get_same, &efct->aux_res, 0);
	if (rc)
		goto err;

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efct_fini_debugfs_aux_dev(efct);
	return rc;
}
#endif

/* Remove a debugfs directory.
 * This removes the named parameter-files and sym-links from the
 * directory, and the directory itself.  It does not do any recursion
 * to subdirectories.
 */
static void efct_fini_debugfs_dir(struct dentry *dir,
				  struct efct_debugfs_parameter *params,
				  const char *const *symlink_names)
{
	if (!dir)
		return;

	while (params && params->name) {
		efct_fini_debugfs_child(dir, params->name);
		params++;
	}
	while (symlink_names && *symlink_names) {
		efct_fini_debugfs_child(dir, *symlink_names);
		symlink_names++;
	}

	debugfs_remove(dir);
}

#ifdef CONFIG_XILINX_AUX_EFCT
void efct_fini_debugfs_aux_dev_client(struct xlnx_efct_client *client)
{
	int i;

	for (i = 0; i < client->efct->rxq_count; i++) {
		efct_fini_debugfs_dir(client->rxq_info[i].debug_dir,
				      efct_debugfs_client_rxq_info_parameters, NULL);
	}
	efct_fini_debugfs_dir(client->debug_dir,
			      efct_debugfs_aux_client_parameters, NULL);
}
#endif

/**
 * efct_init_debugfs_rx_queue - create debugfs directory for RX queue
 * @rx_queue:		efct RX queue
 *
 * Create a debugfs directory containing parameter-files for @rx_queue.
 * Return a negative error code or 0 on success.  The directory must be
 * cleaned up using efct_fini_debugfs_rx_queue().
 */
int efct_init_debugfs_rx_queue(struct efct_rx_queue *rx_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];
	int rc, i = 0;

	/* Create directories */
	if (snprintf(name, sizeof(name), "rxq%d", rx_queue->index)
			>= sizeof(name))
		goto err_len;
	rx_queue->debug_dir = debugfs_create_dir(name,
						 rx_queue->efct->debug_port_dir);
	if (IS_ERR(rx_queue->debug_dir)) {
		rc = PTR_ERR(rx_queue->debug_dir);
		rx_queue->debug_dir = NULL;
		goto err;
	}
	if (!rx_queue->debug_dir)
		goto err_mem;

	rx_queue->nbl.debug_dir = debugfs_create_dir("nbl", rx_queue->debug_dir);
	if (IS_ERR(rx_queue->nbl.debug_dir)) {
		rc = PTR_ERR(rx_queue->nbl.debug_dir);
		rx_queue->nbl.debug_dir = NULL;
		goto err;
	}
	if (!rx_queue->nbl.debug_dir)
		goto err_mem;

	rx_queue->dbl.debug_dir = debugfs_create_dir("dbl", rx_queue->debug_dir);
	if (IS_ERR(rx_queue->dbl.debug_dir)) {
		rc = PTR_ERR(rx_queue->dbl.debug_dir);
		rx_queue->dbl.debug_dir = NULL;
		goto err;
	}
	if (!rx_queue->dbl.debug_dir)
		goto err_mem;

#ifdef CONFIG_XILINX_AUX_EFCT
	rx_queue->hpl.debug_dir = debugfs_create_dir("hpl", rx_queue->debug_dir);
	if (IS_ERR(rx_queue->hpl.debug_dir)) {
		rc = PTR_ERR(rx_queue->hpl.debug_dir);
		rx_queue->hpl.debug_dir = NULL;
		goto err;
	}
	if (!rx_queue->hpl.debug_dir)
		goto err_mem;

	rx_queue->sbl.debug_dir = debugfs_create_dir("sbl", rx_queue->debug_dir);
	if (IS_ERR(rx_queue->sbl.debug_dir)) {
		rc = PTR_ERR(rx_queue->sbl.debug_dir);
		rx_queue->sbl.debug_dir = NULL;
		goto err;
	}
	if (!rx_queue->sbl.debug_dir)
		goto err_mem;
#endif

	/* Create subdirectories for nbl and its files */
	rc = efct_init_debugfs_files(rx_queue->nbl.debug_dir,
				     efct_debugfs_rx_queue_nbl_parameters, 0,
				     efct_debugfs_get_same, &rx_queue->nbl, 0);
	if (rc)
		goto err;

	for (i = 0; i < NIC_BUFFS_PER_QUEUE; i++) {
		if (snprintf(name, sizeof(name), "nb%d", i)
				>= sizeof(name))
			goto err_len;
		rx_queue->nbl.nb[i].debug_dir = debugfs_create_dir(name, rx_queue->nbl.debug_dir);
		if (IS_ERR(rx_queue->nbl.nb[i].debug_dir)) {
			rc = PTR_ERR(rx_queue->nbl.nb[i].debug_dir);
			rx_queue->nbl.nb[i].debug_dir = NULL;
			goto err;
		}
		if (!rx_queue->nbl.nb[i].debug_dir)
			goto err_mem;

		/* Create files */
		rc = efct_init_debugfs_files(rx_queue->nbl.nb[i].debug_dir,
					     efct_debugfs_nbl_parameters, 0, efct_debugfs_get_same,
					     &rx_queue->nbl.nb[i], 0);
		if (rc)
			goto err;
	}

	/* Create subdirectories for nbl and its files */
	for (i = 0; i < RX_MAX_DRIVER_BUFFS; i++) {
		if (snprintf(name, sizeof(name), "db%d", i)
				>= sizeof(name))
			goto err_len;
		rx_queue->dbl.db[i].debug_dir = debugfs_create_dir(name, rx_queue->dbl.debug_dir);
		if (IS_ERR(rx_queue->dbl.db[i].debug_dir)) {
			rc = PTR_ERR(rx_queue->dbl.db[i].debug_dir);
			rx_queue->dbl.db[i].debug_dir = NULL;
			goto err;
		}
		if (!rx_queue->dbl.db[i].debug_dir)
			goto err_mem;

		/* Create files */
		rc = efct_init_debugfs_files(rx_queue->dbl.db[i].debug_dir,
					     efct_debugfs_dbl_parameters, 0, efct_debugfs_get_same,
					     &rx_queue->dbl.db[i], 0);
		if (rc)
			goto err;
	}

#ifdef CONFIG_XILINX_AUX_EFCT
	rc = efct_init_debugfs_files(rx_queue->hpl.debug_dir,
				     efct_debugfs_rx_queue_hpl_parameters, 0,
				     efct_debugfs_get_same, &rx_queue->hpl, 0);
	if (rc)
		goto err;

	/* Create subdirectories for sbl and its files */
	for (i = 0; i < SUPER_BUFFS_PER_QUEUE; i++) {
		if (snprintf(name, sizeof(name), "sb%d", i)
				>= sizeof(name))
			goto err_len;
		rx_queue->sbl.sb[i].debug_dir = debugfs_create_dir(name, rx_queue->sbl.debug_dir);
		if (IS_ERR(rx_queue->sbl.sb[i].debug_dir)) {
			rc = PTR_ERR(rx_queue->sbl.sb[i].debug_dir);
			rx_queue->sbl.sb[i].debug_dir = NULL;
			goto err;
		}
		if (!rx_queue->sbl.sb[i].debug_dir)
			goto err_mem;

		/* Create files */
		rc = efct_init_debugfs_files(rx_queue->sbl.sb[i].debug_dir,
					     efct_debugfs_sbl_parameters, 0, efct_debugfs_get_same,
					     &rx_queue->sbl.sb[i], 0);
		if (rc)
			goto err;
	}
#endif
	rc = efct_init_debugfs_files(rx_queue->debug_dir,
				     efct_debugfs_rx_queue_parameters, 0,
				     efct_debugfs_get_same, rx_queue, 0);
	if (rc)
		goto err;

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efct_fini_debugfs_rx_queue(rx_queue);
	return rc;
}

/**
 * efct_fini_debugfs_rx_queue - remove debugfs directory for RX queue
 * @rx_queue:		efct RX queue
 *
 * Remove directory created for @rx_queue by efct_init_debugfs_rx_queue().
 */
void efct_fini_debugfs_rx_queue(struct efct_rx_queue *rx_queue)
{
	int i = 0;

	/* Remove NBL entries */
	for (i = 0; i < NIC_BUFFS_PER_QUEUE; i++) {
		efct_fini_debugfs_dir(rx_queue->nbl.nb[i].debug_dir,
				      efct_debugfs_nbl_parameters, NULL);
		rx_queue->nbl.nb[i].debug_dir = NULL;
	}
	efct_fini_debugfs_dir(rx_queue->nbl.debug_dir,
			      efct_debugfs_rx_queue_nbl_parameters, NULL);
	rx_queue->nbl.debug_dir = NULL;

	/* Remove DBL entries */
	for (i = 0; i < RX_MAX_DRIVER_BUFFS; i++) {
		efct_fini_debugfs_dir(rx_queue->dbl.db[i].debug_dir,
				      efct_debugfs_dbl_parameters, NULL);
		rx_queue->dbl.db[i].debug_dir = NULL;
	}
	efct_fini_debugfs_dir(rx_queue->dbl.debug_dir, NULL, NULL);
	rx_queue->nbl.debug_dir = NULL;

#ifdef CONFIG_XILINX_AUX_EFCT
	/* Remve HPL entries */
	efct_fini_debugfs_dir(rx_queue->hpl.debug_dir,
			      efct_debugfs_rx_queue_hpl_parameters, NULL);
	rx_queue->hpl.debug_dir = NULL;

	/* Remove SPL Entries */
	for (i = 0; i < SUPER_BUFFS_PER_QUEUE; i++) {
		efct_fini_debugfs_dir(rx_queue->sbl.sb[i].debug_dir,
				      efct_debugfs_sbl_parameters, NULL);
		rx_queue->sbl.sb[i].debug_dir = NULL;
	}
	efct_fini_debugfs_dir(rx_queue->sbl.debug_dir, NULL, NULL);
	rx_queue->sbl.debug_dir = NULL;
#endif
	efct_fini_debugfs_dir(rx_queue->debug_dir,
			      efct_debugfs_rx_queue_parameters, NULL);
	rx_queue->debug_dir = NULL;
}

/**
 * efct_init_debugfs_tx_queue - create debugfs directory for TX queue
 * @tx_queue:		efct TX queue
 *
 * Create a debugfs directory containing parameter-files for @tx_queue.
 * Return a negative error code or 0 on success.  The directory must be
 * cleaned up using efct_fini_debugfs_tx_queue().
 */
int efct_init_debugfs_tx_queue(struct efct_tx_queue *tx_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];
	int rc;

	/* Create directory */
	if (snprintf(name, sizeof(name), "txq%d", tx_queue->txq_index)
			>= sizeof(name))
		goto err_len;
	tx_queue->debug_dir = debugfs_create_dir(name,
						 tx_queue->efct->debug_port_dir);
	if (IS_ERR(tx_queue->debug_dir)) {
		rc = PTR_ERR(tx_queue->debug_dir);
		tx_queue->debug_dir = NULL;
		goto err;
	}
	if (!tx_queue->debug_dir)
		goto err_mem;

	/* Create files */
	rc = efct_init_debugfs_files(tx_queue->debug_dir,
				     efct_debugfs_tx_queue_parameters, 0,
				     efct_debugfs_get_same, tx_queue, 0);
	if (rc)
		goto err;

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efct_fini_debugfs_tx_queue(tx_queue);
	return rc;
}

/**
 * efct_fini_debugfs_tx_queue - remove debugfs directory for TX queue
 * @tx_queue:		efct TX queue
 *
 * Remove directory created for @tx_queue by efct_init_debugfs_tx_queue().
 */
void efct_fini_debugfs_tx_queue(struct efct_tx_queue *tx_queue)
{
	efct_fini_debugfs_dir(tx_queue->debug_dir,
			      efct_debugfs_tx_queue_parameters, NULL);
	tx_queue->debug_dir = NULL;
}

/**
 * efct_init_debugfs_ev_queue - create debugfs directory for EV queue
 * @ev_queue:		efct EV queue
 *
 * Create a debugfs directory containing parameter-files for @ev_queue.
 * Return a negative error code or 0 on success.  The directory must be
 * cleaned up using efct_fini_debugfs_ev_queue().
 */
int efct_init_debugfs_ev_queue(struct efct_ev_queue *ev_queue)
{
	char name[EFCT_DEBUGFS_NAME_LEN];
	int rc;

	/* Create directory */
	if (snprintf(name, sizeof(name), "evq%d", ev_queue->index)
			>= sizeof(name))
		goto err_len;
	ev_queue->debug_dir = debugfs_create_dir(name,
						 ev_queue->efct->debug_port_dir);
	if (IS_ERR(ev_queue->debug_dir)) {
		rc = PTR_ERR(ev_queue->debug_dir);
		ev_queue->debug_dir = NULL;
		goto err;
	}
	if (!ev_queue->debug_dir)
		goto err_mem;

	/* Create files */
	rc = efct_init_debugfs_files(ev_queue->debug_dir,
				     efct_debugfs_ev_queue_parameters, 0,
				     efct_debugfs_get_same, ev_queue, 0);
	if (rc)
		goto err;

	return 0;

 err_len:
	rc = -ENAMETOOLONG;
	goto err;
 err_mem:
	rc = -ENOMEM;
 err:
	efct_fini_debugfs_ev_queue(ev_queue);
	return rc;
}

/**
 * efct_fini_debugfs_ev_queue - remove debugfs directory for ev queue
 * @ev_queue:		efct Event queue
 *
 * Remove directory created for @ev_queue by efct_init_debugfs_ev_queue().
 */
void efct_fini_debugfs_ev_queue(struct efct_ev_queue *ev_queue)
{
	efct_fini_debugfs_dir(ev_queue->debug_dir,
			      efct_debugfs_ev_queue_parameters, NULL);
	ev_queue->debug_dir = NULL;
}

/**
 * efct_fini_debugfs_nic - remove debugfs directories for NIC
 * @efct:		efct NIC
 *
 * Remove debugfs directories created for @efct by efct_init_debugfs_nic().
 */
void efct_fini_debugfs_nic(struct efct_nic *efct)
{
	efct_fini_debugfs_dir(efct->errors.debug_dir,
			      efct_debugfs_nic_error_parameters, NULL);
	efct->errors.debug_dir = NULL;
	efct_fini_debugfs_dir(efct->link_state.debug_dir,
			      efct_debugfs_link_state_parameters, NULL);
	efct->link_state.debug_dir = NULL;
	efct_fini_debugfs_dir(efct->debug_port_dir,
			      efct_debugfs_port_parameters, NULL);
	efct->debug_port_dir = NULL;
}

#ifdef CONFIG_XILINX_AUX_EFCT
/**
 * efct_fini_debugfs_aux_dev - remove debugfs directories for Aux Device
 * @efct:		efct NIC
 *
 * Remove debugfs directories created for @efct by efct_init_debugfs_aux_dev().
 */
void efct_fini_debugfs_aux_dev(struct efct_nic *efct)
{
	efct_fini_debugfs_dir(efct->debug_aux_dir,
			      efct_debugfs_aux_dev_parameters, NULL);
	efct->debug_aux_dir = NULL;
}
#endif
/**
 * efct_fini_debugfs_func - remove debugfs directories for the PCIe func
 * @efct_dev:		pci device
 *
 * Remove debugfs directories created for @efct_dev by efct_init_debugfs_func().
 */
void efct_fini_debugfs_func(struct efct_device *efct_dev)
{
	efct_fini_debugfs_dir(efct_dev->params.debug_dir,
			      efct_debugfs_design_parameters, NULL);
	efct_dev->params.debug_dir = NULL;

	efct_fini_debugfs_dir(efct_dev->debug_dir, efct_debugfs_card_parameters, NULL);
	efct_dev->debug_dir = NULL;
}

/**
 * efct_fini_debugfs - remove debugfs directories for x3 driver
 *
 * Remove directories created by efct_init_debugfs().
 */
void efct_fini_debugfs(void)
{
	debugfs_remove_recursive(efct_debug_root);
	efct_debug_root = NULL;
}
