// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include "net_driver.h"
#include "efct_evq.h"
#include "efct_rx.h"
#include "mcdi.h"
#include "mcdi_functions.h"
#include "efct_reg.h"
#include "bitfield.h"
#include "io.h"

#define TX_NAPI_BUDGET  64
#define RX_NAPI_BUDGET  64

static void efct_update_irq_mod(struct efct_nic *efct, struct efct_ev_queue *evq)
{
	int step = efct->irq_mod_step_ns;

	if (evq->irq_mod_score < evq->irq_adapt_low_thresh) {
		if (evq->irq_moderation_ns > step) {
			evq->irq_moderation_ns -= step;
			efct_mcdi_ev_set_timer(evq, evq->irq_moderation_ns,
					       MC_CMD_SET_EVQ_TMR_IN_TIMER_MODE_INT_HLDOFF, true);
		}
	} else if (evq->irq_mod_score > evq->irq_adapt_high_thresh) {
		if (evq->irq_moderation_ns < efct->irq_rx_moderation_ns) {
			evq->irq_moderation_ns += step;
			efct_mcdi_ev_set_timer(evq, evq->irq_moderation_ns,
					       MC_CMD_SET_EVQ_TMR_IN_TIMER_MODE_INT_HLDOFF, true);
		}
	}

	evq->irq_count = 0;
	evq->irq_mod_score = 0;
}

int efct_nic_event_test_irq_cpu(struct efct_ev_queue *evq)
{
	return READ_ONCE(evq->event_test_cpu);
}

static int efct_process_evq(struct efct_ev_queue *evq, int budget)
{
	struct efct_nic *efct = evq->efct;

	return efct->type->ev_process(evq, budget);
}

void efct_write_evt_prime(struct efct_nic *efct, u32 id, u16 seq)
{
	efct_dword_t hdr;

	EFCT_POPULATE_DWORD_2(hdr, ERF_HZ_READ_IDX, seq,
			      ERF_HZ_EVQ_ID, id);
	efct_writed(&hdr, efct->membase + ER_HZ_PORT0_REG_HOST_EVQ_INT_PRIME);
}

static int efct_poll(struct napi_struct *napi, int budget)
{
	struct efct_ev_queue *evq = container_of(napi, struct efct_ev_queue, napi);
	struct efct_nic *efct = evq->efct;
	int spent;

	if (!budget)
		return 0;

	spent = efct_process_evq(evq, budget);
	if (spent < budget) {
		if (evq->type == EVQ_T_RX && efct->irq_rx_adaptive &&
		    unlikely(++evq->irq_count == evq->irq_adapt_irqs)) {
			efct_update_irq_mod(efct, evq);
		}

		if (napi_complete_done(napi, spent))
			// Host needs to Re-prime or enable the interrupt again
			efct_write_evt_prime(evq->efct, evq->index, evq->consumer_index);
	}
	return spent;
}

int efct_realloc_rx_evqs(struct efct_nic *efct, u32 num_entries)
{
	int i, rc;

	for (i = 0; i < efct->rxq_count; i++) {
		int evq_index = efct->rxq[i].evq_index;

		efct->type->ev_remove(&efct->evq[evq_index]);
		dbl_fini(&efct->rxq[i]);

		efct->rxq[i].num_entries = num_entries;
		efct->rxq[i].num_rx_buffs = num_entries / (DIV_ROUND_UP(efct->rxq[i].buffer_size,
							   efct->rxq[i].pkt_stride));

		rc = dbl_init(&efct->rxq[i]);
		if (rc) {
			netif_err(efct, drv, efct->net_dev,
				  "Failed initialize driver buffer list for rxq %d\n", i);
			goto fail;
		}

		rc = efct->type->ev_probe(efct, evq_index, efct->rxq[i].num_entries);
		if (rc) {
			netif_err(efct, drv, efct->net_dev,
				  "Event queue probe failed, index %d\n", evq_index);
			goto fail;
		}

		set_bit(evq_index, &efct->evq_active_mask);
	}

	return 0;

fail:
	efct->state = STATE_DISABLED;
	return rc;
}

static void efct_init_evq_napi(struct efct_ev_queue *evq)
{
	struct efct_nic *efct = evq->efct;

	evq->napi_dev = efct->net_dev;

	if (evq->type == EVQ_T_RX)
		netif_napi_add(evq->napi_dev, &evq->napi, efct_poll, RX_NAPI_BUDGET);
	else if (evq->type == EVQ_T_TX)
		netif_tx_napi_add(evq->napi_dev, &evq->napi, efct_poll, TX_NAPI_BUDGET);
	else
		evq->napi_dev = NULL;
}

int efct_init_napi(struct efct_nic *efct)
{
	u32 i;

	for (i = 0; i < efct->max_evq_count; ++i)
		efct_init_evq_napi(&efct->evq[i]);

	return 0;
}

static void efct_finish_evq_napi(struct efct_ev_queue *evq)
{
	if (evq->napi_dev)
		netif_napi_del(&evq->napi);
}

void efct_finish_napi(struct efct_nic *efct)
{
	u32 i;

	for (i = 0; i < efct->max_evq_count; ++i)
		efct_finish_evq_napi(&efct->evq[i]);
}

void efct_disable_napi(struct efct_ev_queue *evq)
{
	struct napi_struct *n = &evq->napi;
	unsigned long val, new;

	if (evq->efct->state == STATE_NET_UP) {
		might_sleep();
		set_bit(NAPI_STATE_DISABLE, &n->state);

		for ( ; ; ) {
			val = READ_ONCE(n->state);
			if (val & (NAPIF_STATE_SCHED | NAPIF_STATE_NPSVC)) {
				usleep_range(20, 200);
				continue;
			}

			new = val | NAPIF_STATE_SCHED | NAPIF_STATE_NPSVC;
#if defined(EFCT_HAVE_NET_DEVICE_PREFER_BUSY_POLL)
			new &= ~(NAPIF_STATE_PREFER_BUSY_POLL);
#endif
#if defined(EFCT_HAVE_NET_DEVICE_THREADED) && defined(EFCT_HAVE_NAPI_THREAD)
			new &= ~(NAPIF_STATE_THREADED)
#endif
			if (cmpxchg(&n->state, val, new) == val)
				break;
		}

		hrtimer_cancel(&n->timer);

		clear_bit(NAPI_STATE_DISABLE, &n->state);
	}
}

void efct_enable_napi(struct efct_ev_queue *evq)
{
	struct napi_struct *n = &evq->napi;
	unsigned long val, new;

	if (evq->efct->state == STATE_NET_UP) {
		do {
			val = READ_ONCE(n->state);
			BUG_ON(!test_bit(NAPI_STATE_SCHED, &val));

			new = val & ~(NAPIF_STATE_SCHED | NAPIF_STATE_NPSVC);
#if defined(EFCT_HAVE_NET_DEVICE_THREADED) && defined(EFCT_HAVE_NAPI_THREAD)
			if (n->dev->threaded && n->thread)
				new |= NAPIF_STATE_THREADED;
#endif
		} while (cmpxchg(&n->state, val, new) != val);
		local_bh_disable();
		napi_schedule(&evq->napi);
		local_bh_enable();
	}
}

void efct_synchronize_napi(struct efct_ev_queue *evq)
{
	if (evq->efct->state == STATE_NET_UP)
		napi_synchronize(&evq->napi);
}
