/* SPDX-License-Identifier: GPL-2.0
 ****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_IOCTL_H
#define EFCT_IOCTL_H

#include <linux/sockios.h>
#include "enum.h"

#define EFCT_MCDI_REQUEST2 0xef21
/**
 * struct efct_mcdi_request2 - Parameters for %EFCT_MCDI_REQUEST2 sub-command
 * @cmd: MCDI command type number.
 * @inlen: The length of command parameters, in bytes.
 * @outlen: On entry, the length available for the response, in bytes.
 *	On return, the length used for the response, in bytes.
 * @flags: Flags for the command or response.  The only flag defined
 *	at present is %EFCT_MCDI_REQUEST_ERROR.  If this is set on return,
 *	the MC reported an error.
 * @host_errno: On return, if %EFCT_MCDI_REQUEST_ERROR is included in @flags,
 *	the suggested Linux error code for the error.
 * @payload: On entry, the MCDI command parameters.  On return, the response.
 *
 * If the driver detects invalid parameters or a communication failure
 * with the MC, the ioctl() call will return -1, errno will be set
 * accordingly, and none of the fields will be valid.  If the MC reports
 * an error, the ioctl() call will return 0 but @flags will include the
 * %EFCT_MCDI_REQUEST_ERROR flag.  The MC error code can then be found in
 * @payload (if @outlen was sufficiently large) and a suggested Linux
 * error code can be found in @host_errno.
 *
 * %EFCT_MCDI_REQUEST2 fully supports both MCDIv1 and v2.
 */
struct efct_mcdi_request2 {
	__u16 cmd;
	__u16 inlen;
	__u16 outlen;
	__u16 flags;
	__u32 host_errno;
	/* The maximum payload length is 0x400 (MCDI_CTL_SDU_LEN_MAX_V2) - 4 bytes
	 * = 255 x 32 bit words as MCDI_CTL_SDU_LEN_MAX_V2 doesn't take account of
	 * the space required by the V1 header, which still exists in a V2 command.
	 */
	__u32 payload[255];
};

#define EFCT_MCDI_REQUEST_ERROR	0x0001

/* Get device identity ******************************************************/
#define EFCT_GET_DEVICE_IDS 0xef22
struct efct_device_ids {
	__u16 vendor_id, device_id;		/* PCI device ID */
	__u16 subsys_vendor_id, subsys_device_id; /* PCI subsystem ID */
	__u8 port_num;				/* port number (0-based) */
	__u8 perm_addr[6];			/* non-volatile MAC address */
};

/* For setting the netif carrier on/off *************************************/
#define EFCT_SET_CARRIER 0xef09
struct efct_set_carrier_ioctl {
	__u8 on;
};

/* Testing/QA: set loopback mode ********************************************/
#define EFCT_SET_LOOPBACK 0xef03

/* Parameters for EFCT_SET_LOOPBACK */
struct efct_set_loopback_ioctl {
	/** Loopback mode */
	enum efct_loopback_mode mode;
};

#define LOOPBACK_NEAR   (LOOPBACK_MAX + 1)  /* loopback nearest to bus */
#define LOOPBACK_FAR    (LOOPBACK_MAX + 2)  /* loopback furthest from bus */

/* For setting the PHY power state ******************************************/
#define EFCT_SET_PHY_POWER 0xef0b

struct efct_set_phy_power {
	__u8 on;
};

#ifdef CONFIG_XILINX_PTP
/* Set the NIC-system synchronization status ********************************/
#define EFCT_TS_SET_SYNC_STATUS 0xef27
struct efct_ts_set_sync_status {
	__u32 in_sync;		/* 0 == not in sync, 1 == in sync */
	__u32 timeout;		/* Seconds until no longer in sync */
};
#endif

/* Efx private ioctl number */
/* We do not use the first 3 private ioctls because some utilities expect
 * them to be the old MDIO ioctls.
 */
#define SIOCEFCT (SIOCDEVPRIVATE + 3)

/* Efx private ioctl command structures *************************************/

union efct_ioctl_data {
	struct efct_mcdi_request2 mcdi_request2;
	struct efct_device_ids device_ids;
	struct efct_set_loopback_ioctl set_loopback;
	struct efct_set_carrier_ioctl set_carrier;
	struct efct_set_phy_power set_phy_power;
#ifdef CONFIG_XILINX_PTP
	struct efct_ts_set_sync_status ts_set_sync_status;
#endif
};

/**
 * struct efct_sock_ioctl - Parameters for sfc private ioctl on socket
 * @cmd: Command number
 * @u: Command-specific parameters
 *
 * Usage:
 *     struct ifreq ifr;
 *
 *     struct efct_sock_ioctl efct;
 *
 *     fd = socket(%AF_INET, %SOCK_STREAM, 0);
 *
 *     strncpy(ifr.ifr_name, if_name, %IFNAMSIZ);
 *
 *     ifr.ifr_data = (caddr_t) & efct;
 *
 *     efct.cmd = %EFCT_FROBNOSTICATE;
 *
 *     efct.u.frobnosticate.magic = 42;
 *
 *     ret = ioctl(fd, %SIOCEFCT, & ifr);
 */
struct efct_sock_ioctl {
	/* Command to run */
	__u16 cmd;
	__u16 reserved;
	/* Parameters */
	union efct_ioctl_data u;
} __packed;

#ifdef __KERNEL__
int efct_private_ioctl(struct efct_nic *efct, u16 cmd,
		       union efct_ioctl_data __user *data);
int efct_control_init(void);
void efct_control_fini(void);
#endif /* __KERNEL__ */

#endif /* EFCT_IOCTL_H */
