/* SPDX-License-Identifier: GPL-2.0 */
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_PTP_H
#define EFCT_PTP_H

#include <linux/net_tstamp.h>
#include <linux/pps_kernel.h>
#include <linux/ptp_clock_kernel.h>
#include "net_driver.h"
#include "bitfield.h"

/* Maximum number of events expected to make up a PTP event */
#define	MAX_EVENT_FRAGS			3
/**
 * struct efct_ptp_data - Precision Time Protocol (PTP) state
 * @efct: The NIC context
 * @phcs_node: Node in list of all PHC PTP datas
 * @kref: Reference count.
 * @config: Current timestamp configuration
 * @enabled: PTP operation enabled. If this is disabled normal timestamping
 *	     can still work.
 * @txtstamp: Enable Tx side PTP timestamping
 * @rxtstamp: Enable Rx side PTP timestamping
 * @evt_lock: Lock for manipulating evt_list and evt_free_list
 * @evt_frags: Partly assembled PTP events
 * @evt_frag_idx: Current fragment number
 * @evt_code: Last event code
 * @adapter_base_addr: MAC address of port0 (used as unique identifier) of PHC
 * @mode: Mode in which PTP operating (PTP version)
 * @ns_to_nic_time: Function to convert from scalar nanoseconds to NIC time
 * @nic_to_kernel_time: Function to convert from NIC 32 bit wide second to kernel time
 * @nic64_to_kernel_time: Function to convert from NIC 64 bit wide second to kernel time
 * @capabilities: Capabilities flags from the NIC
 * @rx_ts_inline: Flag for whether RX timestamps are inline (else they are
 *	separate events)
 * @evt_list: List of MC receive events awaiting packets
 * @rx_evts: Instantiated events (on evt_list and evt_free_list)
 * @workwq: Work queue for processing pending PTP operations
 * @work: Work task
 * @reset_required: A serious error has occurred and the PTP task needs to be
 *                  reset (disable, enable).
 * @ts_corrections.ptp_tx: Required driver correction of PTP packet transmit
 *                         timestamps
 * @ts_corrections.ptp_rx: Required driver correction of PTP packet receive
 *                         timestamps
 * @ts_corrections.pps_out: PPS output error (information only)
 * @ts_corrections.pps_in: Required driver correction of PPS input timestamps
 * @ts_corrections.general_tx: Required driver correction of general packet
 *                             transmit timestamps
 * @ts_corrections.general_rx: Required driver correction of general packet
 *                             receive timestamps
 * @nic_time.minor_max: Wrap point for NIC minor times
 * @nic_time.sync_event_diff_min: Minimum acceptable difference between time
 * in packet prefix and last MCDI time sync event i.e. how much earlier than
 * the last sync event time a packet timestamp can be.
 * @nic_time.sync_event_diff_max: Maximum acceptable difference between time
 * in packet prefix and last MCDI time sync event i.e. how much later than
 * the last sync event time a packet timestamp can be.
 * @nic_time.sync_event_minor_shift: Shift required to make minor time from
 * field in MCDI time sync event.
 * @pps_work: pps work task for handling pps events
 * @pps_workwq: pps work queue
 * @phc_clock: Pointer to registered phc device
 * @phc_clock_info: Registration structure for phc device
 * @adjfreq_ppb_shift: Shift required to convert scaled parts-per-billion
 * @pps_data: Data associated with optional HW PPS events
 * @max_adjfreq: Current ppb adjustment, lives here instead of phc_clock_info as
 *		 it must be accessible without PHC support, using private ioctls.
 * @current_adjfreq: Current ppb adjustment.
 * @pin_config: PTP pin functions description
 * @last_delta_valid: Boolean
 * @last_delta: Clock difference between nic and host
 * @host_time_pps: Host time at last PPS
 * @usr_evt_enabled: Flag indicating how NIC generated TS events are handled
 */

struct efct_tx_queue;

struct efct_ptp_data {
	struct efct_nic *efct;
	struct list_head phcs_node;
	struct kref kref;
	struct hwtstamp_config config;
	bool enabled;
	bool txtstamp;
	bool rxtstamp;
	efct_qword_t evt_frags[MAX_EVENT_FRAGS];
	int evt_frag_idx;
	int evt_code;
	u8 serial[EFCT_MAX_VERSION_INFO_LEN];
	void (*ns_to_nic_time)(s64 ns, u32 *nic_major, u32 *nic_minor, u32 *nic_hi);
	ktime_t (*nic_to_kernel_time)(u32 nic_major, u32 nic_minor,
				      s32 correction);
	ktime_t (*nic64_to_kernel_time)(u32 nich, u64 timereg,
					s64 correction);
	u32 capabilities;
	struct {
		s32 ptp_tx;
		s32 ptp_rx;
		s32 pps_out;
		s32 pps_in;
		s32 general_tx;
		s32 general_rx;
	} ts_corrections;
	struct {
		u32 minor_max;
		u32 sync_event_diff_min;
		u32 sync_event_diff_max;
		u32 sync_event_minor_shift;
	} nic_time;
	struct {
		u64 skipped_sync;
		u64 invalid_sync_windows;
		u64 pps_fw;
		u64 pps_hw;
	} sw_stats;
	struct work_struct pps_work;
	struct workqueue_struct *pps_workwq;
	struct ptp_clock *phc_clock;
	struct ptp_clock_info phc_clock_info;
	u32 adjfreq_ppb_shift;
	struct efct_pps_data *pps_data;
	s64 max_adjfreq;
	s64 current_adjfreq;
	struct ptp_pin_desc pin_config[1];
	bool last_delta_valid;
	struct timespec64 last_delta;
	struct pps_event_time host_time_pps;
	u8 usr_evt_enabled;
};

/**
 * struct efct_pps_data - PPS device node informatino
 * @ptp: Pointer to parent ptp structure
 * @s_assert: sys assert time of hw_pps event
 * @n_assert: nic assert time of hw_pps event
 * @s_delta: computed delta between nic and sys clocks
 * @nic_hw_pps_enabled: Are hw_pps events enabled
 * @device: PPS device pointer
 */

struct efct_pps_data {
	struct efct_ptp_data *ptp;
	struct timespec64 s_assert;
	ktime_t n_assert;
	struct timespec64 s_delta;
	bool nic_hw_pps_enabled;
	struct pps_device *device;
	int last_ev;
};

struct efct_ptp_timeset {
	struct timespec64 prets;
	u64 nictime;
	struct timespec64 posts;
	s64 window;	/* Derived: end - start */
	s64 mc_host_diff;	/* Derived: mc_time - host_time */
};

int efct_ptp_probe_setup(struct efct_nic *efct);
void efct_ptp_remove_setup(struct efct_nic *efct);
int efct_ptp_get_ts_config(struct net_device *net_dev, struct ifreq *ifr);
int efct_ptp_set_ts_config(struct net_device *net_dev, struct ifreq *ifr);
int efct_ptp_enable_ts(struct efct_nic *efct, struct hwtstamp_config *init);
void efct_ptp_event(struct efct_nic *efct, efct_qword_t *ev);
int efct_ethtool_get_ts_info(struct net_device *net_dev, struct ethtool_ts_info *ts_info);
void efct_ptp_get_ts_info(struct efct_nic *efct, struct ethtool_ts_info *ts_info);
int efct_ptp_ts_set_sync_status(struct efct_nic *efct, u32 in_sync, u32 timeout);
void efct_include_ts_in_skb(struct efct_tx_queue *txq, u64 partial_ts, struct sk_buff *skb);
int efct_ptp_tx_ts_event(struct efct_nic *efct, bool flag);
void efct_ptp_reset_stats(struct efct_nic *efct);
size_t efct_ptp_describe_stats(struct efct_nic *efct, u8 *strings);
size_t efct_ptp_update_stats(struct efct_nic *efct, u64 *stats);
int efct_ptp_subscribe_timesync(struct efct_ev_queue *eventq);
int efct_ptp_unsubscribe_timesync(struct efct_ev_queue *eventq);
void efct_ptp_evt_data_init(struct efct_nic *efct);
int efct_ptp_stop(struct efct_nic *efct);
int efct_ptp_start(struct efct_nic *efct);
int efct_ptp_hw_pps_enable(struct efct_nic *efct, bool enable);
bool efct_phc_exposed(struct efct_nic *efct);

#endif
