/* SPDX-License-Identifier: GPL-2.0
 ****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_ENUM_H
#define EFCT_ENUM_H

/**
 * enum efct_loopback_mode - loopback modes
 * @LOOPBACK_NONE: no loopback
 * @LOOPBACK_DATA: data path loopback
 * @LOOPBACK_GMAC: loopback within GMAC
 * @LOOPBACK_XGMII: loopback after XMAC
 * @LOOPBACK_XGXS: loopback within BPX after XGXS
 * @LOOPBACK_XAUI: loopback within BPX before XAUI serdes
 * @LOOPBACK_GMII: loopback within BPX after GMAC
 * @LOOPBACK_SGMII: loopback within BPX within SGMII
 * @LOOPBACK_XGBR: loopback within BPX within XGBR
 * @LOOPBACK_XFI: loopback within BPX before XFI serdes
 * @LOOPBACK_XAUI_FAR: loopback within BPX after XAUI serdes
 * @LOOPBACK_GMII_FAR: loopback within BPX before SGMII
 * @LOOPBACK_SGMII_FAR: loopback within BPX after SGMII
 * @LOOPBACK_XFI_FAR: loopback after XFI serdes
 * @LOOPBACK_GPHY: loopback within 1G PHY at unspecified level
 * @LOOPBACK_PHYXS: loopback within 10G PHY at PHYXS level
 * @LOOPBACK_PCS: loopback within 10G PHY at PCS level
 * @LOOPBACK_PMAPMD: loopback within 10G PHY at PMAPMD level
 * @LOOPBACK_XPORT: cross port loopback
 * @LOOPBACK_XGMII_WS: wireside loopback excluding XMAC
 * @LOOPBACK_XAUI_WS: wireside loopback within BPX within XAUI serdes
 * @LOOPBACK_XAUI_WS_FAR: wireside loopback within BPX including XAUI serdes
 * @LOOPBACK_XAUI_WS_NEAR: wireside loopback within BPX excluding XAUI serdes
 * @LOOPBACK_GMII_WS: wireside loopback excluding GMAC
 * @LOOPBACK_XFI_WS: wireside loopback excluding XFI serdes
 * @LOOPBACK_XFI_WS_FAR: wireside loopback including XFI serdes
 * @LOOPBACK_PHYXS_WS: wireside loopback within 10G PHY at PHYXS level
 */
/* Please keep up-to-date w.r.t the following two #defines */
enum efct_loopback_mode {
	LOOPBACK_NONE = 0,
	LOOPBACK_DATA = 1,
	LOOPBACK_GMAC = 2,
	LOOPBACK_XGMII = 3,
	LOOPBACK_XGXS = 4,
	LOOPBACK_XAUI = 5,
	LOOPBACK_GMII = 6,
	LOOPBACK_SGMII = 7,
	LOOPBACK_XGBR = 8,
	LOOPBACK_XFI = 9,
	LOOPBACK_XAUI_FAR = 10,
	LOOPBACK_GMII_FAR = 11,
	LOOPBACK_SGMII_FAR = 12,
	LOOPBACK_XFI_FAR = 13,
	LOOPBACK_GPHY = 14,
	LOOPBACK_PHYXS = 15,
	LOOPBACK_PCS = 16,
	LOOPBACK_PMAPMD = 17,
	LOOPBACK_XPORT = 18,
	LOOPBACK_XGMII_WS = 19,
	LOOPBACK_XAUI_WS = 20,
	LOOPBACK_XAUI_WS_FAR = 21,
	LOOPBACK_XAUI_WS_NEAR = 22,
	LOOPBACK_GMII_WS = 23,
	LOOPBACK_XFI_WS = 24,
	LOOPBACK_XFI_WS_FAR = 25,
	LOOPBACK_PHYXS_WS = 26,
	LOOPBACK_PMA_INT = 27,
	LOOPBACK_SD_NEAR = 28,
	LOOPBACK_SD_FAR = 29,
	LOOPBACK_PMA_INT_WS = 30,
	LOOPBACK_SD_FEP2_WS = 31,
	LOOPBACK_SD_FEP1_5_WS = 32,
	LOOPBACK_SD_FEP_WS = 33,
	LOOPBACK_SD_FES_WS = 34,
	LOOPBACK_AOE_INT_NEAR = 35,
	LOOPBACK_DATA_WS = 36,
	LOOPBACK_FORCE_EXT_LINK = 37,
	LOOPBACK_MAX
};

#define LOOPBACK_TEST_MAX LOOPBACK_PMAPMD

enum efct_sync_events_state {
	SYNC_EVENTS_DISABLED = 0,
	SYNC_EVENTS_REQUESTED,
	SYNC_EVENTS_VALID,
};

/* These loopbacks occur within the controller */
#define LOOPBACKS_INTERNAL ((1 << LOOPBACK_DATA) |		\
			    (1 << LOOPBACK_GMAC) |		\
			    (1 << LOOPBACK_XGMII) |		\
			    (1 << LOOPBACK_XGXS) |		\
			    (1 << LOOPBACK_XAUI) |		\
			    (1 << LOOPBACK_GMII) |		\
			    (1 << LOOPBACK_SGMII) |		\
			    (1 << LOOPBACK_SGMII) |		\
			    (1 << LOOPBACK_XGBR) |		\
			    (1 << LOOPBACK_XFI) |		\
			    (1 << LOOPBACK_XAUI_FAR) |		\
			    (1 << LOOPBACK_GMII_FAR) |		\
			    (1 << LOOPBACK_SGMII_FAR) |		\
			    (1 << LOOPBACK_XFI_FAR) |		\
			    (1 << LOOPBACK_XGMII_WS) |		\
			    (1 << LOOPBACK_XAUI_WS) |		\
			    (1 << LOOPBACK_XAUI_WS_FAR) |	\
			    (1 << LOOPBACK_XAUI_WS_NEAR) |	\
			    (1 << LOOPBACK_GMII_WS) |		\
			    (1 << LOOPBACK_XFI_WS) |		\
			    (1 << LOOPBACK_XFI_WS_FAR))

#define LOOPBACKS_WS ((1 << LOOPBACK_XGMII_WS) |		\
		      (1 << LOOPBACK_XAUI_WS) |			\
		      (1 << LOOPBACK_XAUI_WS_FAR) |		\
		      (1 << LOOPBACK_XAUI_WS_NEAR) |		\
		      (1 << LOOPBACK_GMII_WS) |			\
		      (1 << LOOPBACK_XFI_WS) |			\
		      (1 << LOOPBACK_XFI_WS_FAR) |		\
		      (1 << LOOPBACK_PHYXS_WS))

#define LOOPBACKS_EXTERNAL(_efct)					\
	((_efct)->loopback_modes & ~LOOPBACKS_INTERNAL &			\
	 ~(1 << LOOPBACK_NONE))

#define LOOPBACK_MASK(_efct) \
	(1 << (_efct)->loopback_mode)

#define LOOPBACK_INTERNAL(_efct) \
	(!!(LOOPBACKS_INTERNAL & LOOPBACK_MASK(_efct)))

#define LOOPBACK_EXTERNAL(_efct) \
	(!!(LOOPBACK_MASK(_efct) & LOOPBACKS_EXTERNAL(_efct)))

#define LOOPBACK_CHANGED(_from, _to, _mask) \
	(!!((LOOPBACK_MASK(_from) ^ LOOPBACK_MASK(_to)) & (_mask)))

#define LOOPBACK_OUT_OF(_from, _to, _mask) \
	((LOOPBACK_MASK(_from) & (_mask)) && !(LOOPBACK_MASK(_to) & (_mask)))

/*****************************************************************************/

/**
 * enum reset_type - reset types
 *
 * %RESET_TYPE_ALL, %RESET_TYPE_WORLD and
 * %RESET_TYPE_DISABLE specify the method/scope of the reset.  The
 * other valuesspecify reasons, which efct_schedule_reset() will choose
 * a method for.
 *
 * Reset methods are numbered in order of increasing scope.
 *
 * @RESET_TYPE_ALL: Reset datapath, MAC and PHY
 * @RESET_TYPE_WORLD: Reset as much as possible
 * unsuccessful.
 * @RESET_TYPE_DATAPATH: Reset datapath only.
 * @RESET_TYPE_MC_BIST: MC entering BIST mode.
 * @RESET_TYPE_DISABLE: Reset datapath, MAC and PHY; leave NIC disabled
 * @RESET_TYPE_TX_WATCHDOG: reset due to TX watchdog
 * @RESET_TYPE_MC_FAILURE: MC reboot/assertion
 * @RESET_TYPE_MCDI_TIMEOUT: MCDI timeout.
 */
enum reset_type {
	RESET_TYPE_ALL,
	RESET_TYPE_WORLD,
	RESET_TYPE_DATAPATH,
	RESET_TYPE_MC_BIST,
	RESET_TYPE_DISABLE,
	RESET_TYPE_MAX_METHOD,
	RESET_TYPE_TX_WATCHDOG,
	RESET_TYPE_MC_FAILURE,
	/* RESET_TYPE_MCDI_TIMEOUT is actually a method, not
	 * a reason, but it doesn't fit the scope hierarchy (it's not well-
	 * ordered by inclusion)
	 * We encode this by having its enum values be greater than
	 * RESET_TYPE_MAX_METHOD.  This also prevents issuing it with
	 * efct_ioctl_reset
	 */
	RESET_TYPE_MCDI_TIMEOUT,
	RESET_TYPE_MAX,
};

#endif /* EFCT_ENUM_H */
