// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef EFCT_KERNEL_COMPAT_H
#define EFCT_KERNEL_COMPAT_H

#include <linux/pci.h>
#include <linux/time.h>
#include <linux/skbuff.h>
#include <linux/netdevice.h>
#include <linux/etherdevice.h>


/**************************************************************************
 *
 * Autoconf compatibility
 *
 **************************************************************************/

#include "autocompat.h"

#ifndef USER_TICK_USEC
#define USER_TICK_USEC TICK_USEC
#endif

#ifndef pci_info
#define pci_info(pdev, fmt, arg...)	dev_info(&(pdev)->dev, fmt, ##arg)
#endif
#ifndef pci_warn
#define pci_warn(pdev, fmt, arg...)	dev_warn(&(pdev)->dev, fmt, ##arg)
#endif
#ifndef pci_err
#define pci_err(pdev, fmt, arg...)	dev_err(&(pdev)->dev, fmt, ##arg)
#endif
#ifndef pci_dbg
#define pci_dbg(pdev, fmt, arg...)	dev_dbg(&(pdev)->dev, fmt, ##arg)
#endif

#ifdef EFCT_HAVE_RTC_TIME64
	#define rtc_time64_to_tm rtc_time_to_tm
#endif

#ifndef EFCT_HAVE_EEPROM_BY_PAGE
#define ETH_MODULE_EEPROM_PAGE_LEN	128
#endif

#ifndef EFCT_HAVE_PPM_TO_PPB
/**
 * scaled_ppm_to_ppb() - convert scaled ppm to ppb
 *
 * @ppm:    Parts per million, but with a 16 bit binary fractional field
 */
static inline long scaled_ppm_to_ppb(long ppm)
{
	/*
	 * The 'freq' field in the 'struct timex' is in parts per
	 * million, but with a 16 bit binary fractional field.
	 *
	 * We want to calculate
	 *
	 *    ppb = scaled_ppm * 1000 / 2^16
	 *
	 * which simplifies to
	 *
	 *    ppb = scaled_ppm * 125 / 2^13
	 */
	s64 ppb = 1 + ppm;

	ppb *= 125;
	ppb >>= 13;
	return (long)ppb;
}
#endif

#ifdef EFX_NEED_TIMESPEC64_TO_NS_SIGNED

#define KTIME_MIN	(-KTIME_MAX - 1)

#define KTIME_SEC_MIN	(KTIME_MIN / NSEC_PER_SEC)

static inline s64 efct_timespec64_to_ns(const struct timespec64 *ts)

{

	/* Prevent multiplication overflow / underflow */

	if (ts->tv_sec >= KTIME_SEC_MAX)

		return KTIME_MAX;

	if (ts->tv_sec <= KTIME_SEC_MIN)

		return KTIME_MIN;

	return ((s64) ts->tv_sec * NSEC_PER_SEC) + ts->tv_nsec;

}

#define timespec64_to_ns efct_timespec64_to_ns

#endif

#ifndef EFCT_HAVE_ETH_HW_ADDR_SET
static inline void eth_hw_addr_set(struct net_device *dev, const u8 *addr)
{
	ether_addr_copy(dev->dev_addr, addr);
}
#endif

#ifdef EFCT_NEED_SKB_FRAG_OFF
/**
 * skb_frag_off() - Returns the offset of a skb fragment
 * @frag: the paged fragment
 */
static inline u32 skb_frag_off(const skb_frag_t *frag)
{
	/* This later got renamed bv_offset (because skb_frag_t is now really
	 * a struct bio_vec), but the page_offset name should work in any
	 * kernel that doesn't already have skb_frag_off defined.
	 */
	return frag->page_offset;
}
#endif

#ifdef EFCT_HAVE_OLD_DEV_OPEN
static inline int efct_dev_open(struct net_device *netdev, void *unused __always_unused)
{
	return dev_open(netdev);
}

#define dev_open efct_dev_open
#endif

#ifndef EFCT_HAVE_NETDEV_XMIT_MORE
#ifdef EFCT_HAVE_SKB_XMIT_MORE
/* This relies on places that use netdev_xmit_more having an SKB structure
 * called skb.
 */
#define netdev_xmit_more()  (skb->xmit_more)
#else
#define netdev_xmit_more()  (0)
#endif
#endif

#ifdef EFCT_NEED_BYTE_QUEUE_LIMITS
static inline void netdev_tx_sent_queue(struct netdev_queue *dev_queue,
					u32 bytes)
{}
static inline void netdev_tx_completed_queue(struct netdev_queue *dev_queue,
					     u32 pkts, u32 bytes)
{}
static inline void netdev_tx_reset_queue(struct netdev_queue *q) {}
#endif

#ifdef EFCT_HAVE_NETDEV_REGISTER_RH
/* The _rh versions of these appear in RHEL7.
 * Wrap them to make the calling code simpler.
 */
static inline int efct_register_netdevice_notifier(struct notifier_block *b)
{
	return register_netdevice_notifier_rh(b);
}

static inline int efct_unregister_netdevice_notifier(struct notifier_block *b)
{
	return unregister_netdevice_notifier_rh(b);
}

#define register_netdevice_notifier efct_register_netdevice_notifier
#define unregister_netdevice_notifier efct_unregister_netdevice_notifier
#endif

#if defined(EFCT_NEED_HWMON_DEVICE_REGISTER_WITH_INFO)
struct hwmon_chip_info;
struct attribute_group;

enum hwmon_sensor_types {
	hwmon_chip,
	hwmon_temp,
	hwmon_in,
	hwmon_curr,
	hwmon_power,
	hwmon_energy,
	hwmon_humidity,
	hwmon_fan,
	hwmon_pwm,
};

#ifdef EFCT_HAVE_HWMON_CLASS_DEVICE
#define EFCT_HWMON_DEVICE_REGISTER_TYPE class_device
#else
#define EFCT_HWMON_DEVICE_REGISTER_TYPE device
#endif

struct EFCT_HWMON_DEVICE_REGISTER_TYPE *hwmon_device_register_with_info(struct device *dev,
								       const char *name __always_unused,
								       void *drvdata __always_unused,
								       const struct hwmon_chip_info *info __always_unused,
								       const struct attribute_group **extra_groups __always_unused);

#else
#if defined(EFCT_NEED_HWMON_T_ALARM)
#define HWMON_T_ALARM	BIT(hwmon_temp_alarm)
#endif
#endif
/* For RHEL 6 and 7 the above takes care of hwmon_device_register_with_info(),
 * but they are missing the read_string() API in struct hwmon_ops.
 */
#if defined(HWMON_T_MIN) && (defined(EFCT_HAVE_HWMON_READ_STRING) ||	\
			     defined(EFCT_HAVE_HWMON_READ_STRING_CONST))
#define EFCT_HAVE_HWMON_DEVICE_REGISTER_WITH_INFO
#endif

#if defined(EFCT_HAVE_NET_DEVLINK_H) && defined(EFCT_HAVE_NDO_GET_DEVLINK) && defined(EFCT_HAVE_DEVLINK_INFO) && defined(CONFIG_NET_DEVLINK)
/* Minimum requirements met to use the kernel's devlink support */
#include <net/devlink.h>

/* devlink is available, augment the provided support with wrappers and stubs
 * for newer APIs as appropriate.
 */
#define EFCT_USE_DEVLINK

#ifdef EFCT_NEED_DEVLINK_INFO_BOARD_SERIAL_NUMBER_PUT
static inline int devlink_info_board_serial_number_put(struct devlink_info_req *req,
						       const char *bsn)
{
	/* Do nothing */
	return 0;
}
#endif
#ifdef EFCT_NEED_DEVLINK_FLASH_UPDATE_TIMEOUT_NOTIFY
void devlink_flash_update_timeout_notify(struct devlink *devlink,
					 const char *status_msg,
					 const char *component,
					 unsigned long timeout);
#endif

#else

/* devlink is not available, provide a 'fake' devlink info request structure
 * and functions to expose the version information via a file in sysfs.
 */

struct devlink_info_req {
	char *buf;
	size_t bufsize;
};

int devlink_info_serial_number_put(struct devlink_info_req *req, const char *sn);
int devlink_info_driver_name_put(struct devlink_info_req *req, const char *name);
int devlink_info_board_serial_number_put(struct devlink_info_req *req,
					 const char *bsn);
int devlink_info_version_fixed_put(struct devlink_info_req *req,
				   const char *version_name,
				   const char *version_value);
int devlink_info_version_stored_put(struct devlink_info_req *req,
				    const char *version_name,
				    const char *version_value);
int devlink_info_version_running_put(struct devlink_info_req *req,
				     const char *version_name,
				     const char *version_value);

/* Provide a do-nothing stubs for the flash update status notifications */

struct devlink;

static inline void devlink_flash_update_status_notify(struct devlink *devlink,
						      const char *status_msg,
						      const char *component,
						      unsigned long done,
						      unsigned long total)
{
	/* Do nothing */
}

static inline void devlink_flash_update_timeout_notify(struct devlink *devlink,
						       const char *status_msg,
						       const char *component,
						       unsigned long timeout)
{
	/* Do nothing */
}

#endif	/* EFCT_HAVE_NET_DEVLINK_H && EFCT_HAVE_NDO_GET_DEVLINK &&
 * EFCT_HAVE_DEVLINK_INFO && CONFIG_NET_DEVLINK
 */

/* Irrespective of whether devlink is available, use the generic devlink info
 * version object names where possible.  Many of these definitions were added
 * to net/devlink.h over time so augment whatever is provided.
 */
#ifndef DEVLINK_INFO_VERSION_GENERIC_BOARD_ID
#define DEVLINK_INFO_VERSION_GENERIC_BOARD_ID		"board.id"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_BOARD_REV
#define DEVLINK_INFO_VERSION_GENERIC_BOARD_REV		"board.rev"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_BOARD_MANUFACTURE
#define DEVLINK_INFO_VERSION_GENERIC_BOARD_MANUFACTURE	"board.manufacture"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_ASIC_ID
#define DEVLINK_INFO_VERSION_GENERIC_ASIC_ID		"asic.id"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_ASIC_REV
#define DEVLINK_INFO_VERSION_GENERIC_ASIC_REV		"asic.rev"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW
#define DEVLINK_INFO_VERSION_GENERIC_FW			"fw"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_MGMT
#define DEVLINK_INFO_VERSION_GENERIC_FW_MGMT		"fw.mgmt"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_MGMT_API
#define DEVLINK_INFO_VERSION_GENERIC_FW_MGMT_API	"fw.mgmt.api"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_APP
#define DEVLINK_INFO_VERSION_GENERIC_FW_APP		"fw.app"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_UNDI
#define DEVLINK_INFO_VERSION_GENERIC_FW_UNDI		"fw.undi"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_NCSI
#define DEVLINK_INFO_VERSION_GENERIC_FW_NCSI		"fw.ncsi"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_PSID
#define DEVLINK_INFO_VERSION_GENERIC_FW_PSID		"fw.psid"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_ROCE
#define DEVLINK_INFO_VERSION_GENERIC_FW_ROCE		"fw.roce"
#endif
#ifndef DEVLINK_INFO_VERSION_GENERIC_FW_BUNDLE_ID
#define DEVLINK_INFO_VERSION_GENERIC_FW_BUNDLE_ID	"fw.bundle_id"
#endif

#endif /* EFCT_KERNEL_COMPAT_H */
