#!/bin/bash

shopt -s nullglob

allargs="$@"
p=$(basename "$0")
bin=$(dirname "$0")

err  () { echo 1>&2 "$*";   }
log  () { err "$p: $*";     }
fail () { log "$*"; exit 1; }
try  () { "$@" || fail "ERROR: '$*' failed"; }

DIR=$(dirname "$0")

N_MODS="xilinx_efct"

NET_OPT=

usage () {
  err
  err "bad command line for linux_net/load.sh '$allargs'"
  err
  err "usage:  $p [options]"
  err
  err "configuration options"
  err "  -nounload              - don't unload the driver if already loaded"
  err "  -noifup                - don't bring the interfaces up"
  err "  -mtu <mtu>             - Set interface mtu to <mtu>"
  err "  -auxrepo <path>        - Aux driver repository path"
  err "  -netparm  <parameter>  - Set module parameters"

  err
  err "debug options"
  err "  --debug         - verbose logging of commands in this script"
  err
  exit 1
}

trylog() {
  tmpf=$(mktemp /tmp/tmp.XXXXXX)
  "$@" &>$tmpf
  rc=$?
  [ $rc = 0 ] || {
    cat $tmpf
    rm -f $tmpf
    fail "FAILED: $* (rc $rc)"
  }
  rm -f $tmpf
  return $rc
}

###############################################################################
# insert module for net device driver

# A mount point has a different block device from its parent.
# (Unless it's the root directory, but we can ignore that here.)
is_mount_point () {
  test -d "$1" -a "$(stat -c %D "$1" 2>/dev/null)" != "$(stat -c %D "$1/.." 2>/dev/null)"
}


# NetworkManager runs DHCP on link-up and removes IP addrs on failure
# Match runbench prepare_machine.py::verify_dhcp_disabled()
disable_dhcp () {
  local intf="$1"
  local cfg_file="/etc/sysconfig/network-scripts/ifcfg-$intf"
  local using_nm=false


  (service NetworkManager status 2>&1 | egrep -qi '(started|running|active)') && using_nm=true

  test -d /etc/sysconfig/network-scripts || return
  if [ ! -f "$cfg_file" ]; then
     $using_nm || return # without NetworkManager, no cfgfile => unmanaged
     touch $cfg_file
  fi

  cp $cfg_file $cfg_file.new
  sed -nie '/DEVICE=/ n; /BOOTPROTO=/ n; /HWADDR=/ n; /ONBOOT/ n; /NM_CONTROLLED/ n; p' $cfg_file.new
  local hwaddr=$(cat /sys/class/net/$intf/address)

  cat >> $cfg_file.new <<EOF
DEVICE=$intf
BOOTPROTO=static
HWADDR=$hwaddr
ONBOOT=no
NM_CONTROLLED=no
EOF

  if ! cmp -s $cfg_file $cfg_file.new; then
    echo "Writing ifcfg-$intf then sleep 5 to allow NetworkManager to detect it  ..."
    mv $cfg_file.new $cfg_file
    sleep 5

    # bug 22617 - DHCP may have already started. NB dhclient is for RHEL
    if pkill -f "dhclient-$intf"; then
      echo "killing dhclient-$intf ..."
      sleep 2
      # Killing dhclient has been seen to leave the interface down
      ip link set dev $intf up
    fi
  fi
}

doaux () {
  kver=`uname -r`
  if [ -f "/boot/config-${kver}" ]; then
    if grep -q '^CONFIG_AUXILIARY_BUS=y' /boot/config-"${kver}"; then
       echo "Auxiliary bus statically compiled in kernel"
       return
    fi
  fi

  if lsmod | grep -q '^auxiliary ';
  then
    return
  fi

  if [ "${auxrepo}" ]; then
    if [ -f "${auxrepo}/drivers/base/auxiliary.ko" ]; then
      trylog /sbin/insmod "${auxrepo}"/drivers/base/auxiliary.ko
    else
      fail "No auxiliary.ko found at ${auxrepo}"
    fi
  elif /sbin/modprobe -q auxiliary; then
      return
  elif [ -f "${DIR}/../../../../../cns-auxiliary-bus/drivers/base/auxiliary.ko" ]; then
      trylog /sbin/insmod "${DIR}"/../../../../../cns-auxiliary-bus/drivers/base/auxiliary.ko
  else
      fail "No auxiliary.ko found"
  fi
}

donet () {
  PATH=/sbin:/usr/sbin:/bin:$PATH
  if /sbin/lsmod | grep -q "^\(${N_MODS/ /\|}\)\>"; then
    echo "Module already loaded."
  else
    for m in $N_MODS; do
	test -f $DIR/$m.ko || continue
	# Only pass known options to the modules
	modinfo -F parm $DIR/$m.ko > /tmp/parm.$$
	echo "dyndbg: Dynamic module debugging" >> /tmp/parm.$$
	MOD_OPT=
	for option in $NET_OPT;
	do
	  grep -q "^${option%%=*}:" /tmp/parm.$$
	  if [ $? -eq 0 ];
	  then
	      MOD_OPT="$MOD_OPT $option";
	  else
	      echo "Module $m has no option ${option%%=*}, ignoring it";
	  fi
	done
	test -d /sys/module/$(basename $m) && continue
        trylog /sbin/insmod $DIR/$m.ko $MOD_OPT
    done
    rm -f /tmp/parm.$$
  fi

  # Find all interfaces created by driver
  declare -a interfaces
  for d in /sys/class/net/*; do
    driver="$(readlink "$d"/device/driver 2>/dev/null || readlink "$d"/driver)"
    if echo "$driver" | grep -qE "/${N_MODS// /$|}$"; then
      interfaces[${#interfaces[*]}]="$(basename "$d")"
    fi
  done
  if [ ${#interfaces[*]} -eq 0 ]; then
    if [ -z "`lspci -d 10ee:`" ] && [ -z "`lspci -d 10ee:5084`" ]; then
      fail "no Xilinx NICs detected in this machine"
    else
      fail "driver failed to create any interfaces"
    fi
  fi

  # Sort by MAC address to give stable IPs (and match runbench behaviour)
  # Include driver name so that we can filter on the driver
  intf_by_mac="$(for i in "${interfaces[@]}"; do echo "$(cat /sys/class/net/$i/address) $i $(basename $(readlink /sys/class/net/$i/device/driver))"; done | sort)"
  interfaces=($(echo "$intf_by_mac" | cut -d ' ' -f 2))

  # Sanity check that all interfaces have unique globally-assigned MAC addresses
  mac_dup="$(echo "$intf_by_mac" | cut -d ' ' -f 1 | uniq -d)"
  [ -z "${mac_dup}" ] || fail "Repeated MAC address(es) ${mac_dup}"

  # Find driver debug dir
  if grep -q debugfs /proc/filesystems; then
    if is_mount_point /debug; then
      driver_debug_dir=/debug/xilinx_efct
    else
      driver_debug_dir=/sys/kernel/debug/xilinx_efct
      if ! is_mount_point /sys/kernel/debug; then
        mount -t debugfs debugfs /sys/kernel/debug
      fi
    fi
  fi
  if ! [ -d "$driver_debug_dir" ]; then
    echo "Warning: xilinx_efct driver debug dir not found"
  fi

  if [ $NO_IFUP -eq 0 ] ; then
      # bring up interfaces
      for ethif in "${interfaces[@]}"; do
        disable_dhcp $ethif

        # Set an IP address using DNS for <host>-l/-m
        echo Bring up interface $ethif

	# Warn if there is a config file in place (might use DHCP and change routing on failure)
	ifcfg="/etc/sysconfig/network-scripts/ifcfg-$ethif"
	if [ -f "$ifcfg" ]; then
	    err "WARNING network interface configuration exists. Perhaps remove '$ifcfg'"
	fi

        /sbin/ip link set $ethif up mtu $mtu

      done
  fi

  # Warn if the PCI-X chipset does not support MSI
  (lspci | grep "8131" > /dev/null) && {
    lspci -d 1924: -vvv 2>/dev/null | grep -qi express ||
      err "WARNING: AMD 8131 PCI-X tunnel present. MSI for PCI-X cards will" \
          "be broken!"
  }

  # Check that the kernel is configured for MSI
  kver=`uname -r`
  if [ -f "/boot/*/config-$kver" ]; then
    grep -qs 'CONFIG_PCI_MSI=y' /boot/*/config-${kver} || \
      err "MSI support might not be compiled in this kernel"
  fi
  if [ -f "/boot/config-$kver" ]; then
    grep -qs 'CONFIG_PCI_MSI=y' /boot/config-${kver} || \
      err "MSI support might not be compiled in this kernel"
  fi
  if [ -f "/proc/kallsyms" ]; then
    grep -qs pci_enable_msi /proc/kallsyms || \
      err "Running kernel does not appear to support MSI"
  fi

}


###############################################################################
# main()

[ `whoami` == "root" ] || fail "Please run as root"

NO_IFUP=0
mtu=1500
unload=true
auxrepo=""
while [ $# -gt 0 ]; do
  case "$1" in
    -nounload)  unload=false;;
    -noifup)    NO_IFUP=1;;
    -mtu)	mtu="$2"; shift;;
    -auxrepo)	auxrepo="$2"; shift;;
    -netparm)	NET_OPT="$NET_OPT $2"; shift;;
    --debug)	set -x;;
    -*)		usage;;
    *)		break;;
  esac
  shift
done

[ $# -gt 1 ] && usage

# By default we unload before loading, to ensure that the user gets the
# driver they're expecting.  (This is important because some kernels
# include a version of our sfc driver, so we need to remove it by default).
$unload && { "$bin/unload.sh" || exit; }

#
# Add NIC / driver specific options
NET_OPT="$NET_OPT"

doaux
donet

exit 0
