// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */
#include <linux/delay.h>
#include <linux/moduleparam.h>
#include "net_driver.h"
#include "nic.h"
#include "efct_common.h"
#include "io.h"
#include "mcdi_pcol.h"
#ifdef CONFIG_XILINX_PTP
#include "efct_ptp.h"
#endif
struct efct_mcdi_copy_buffer {
	_MCDI_DECLARE_BUF(buffer, MCDI_CTL_SDU_LEN_MAX);
};

/**************************************************************************
 *
 * Management-Controller-to-Driver Interface
 *
 **************************************************************************
 */

/* Default RPC timeout for NIC types that don't specify. */
#define MCDI_RPC_TIMEOUT	(10 * HZ)
/* Timeout for acquiring the bus; there may be multiple outstanding requests. */
#define MCDI_ACQUIRE_TIMEOUT	(MCDI_RPC_TIMEOUT * 5)
/* Timeout waiting for a command to be authorised */
#define MCDI_PROXY_TIMEOUT	(10 * HZ)

#ifdef CONFIG_XILINX_MCDI_LOGGING
/* printk has this internal limit. Taken from printk.c. */
#define LOG_LINE_MAX		(1024 - 32)
#endif

/* A reboot/assertion causes the MCDI status word to be set after the
 * command word is set or a REBOOT event is sent. If we notice a reboot
 * via these mechanisms then wait 250ms for the status word to be set.
 */
#define MCDI_STATUS_DELAY_US		100
#define MCDI_STATUS_DELAY_COUNT		2500
#define MCDI_STATUS_SLEEP_MS						\
	(MCDI_STATUS_DELAY_US * MCDI_STATUS_DELAY_COUNT / 1000)

#ifdef CONFIG_XILINX_MCDI_LOGGING
static bool mcdi_logging_default;
module_param(mcdi_logging_default, bool, 0644);
MODULE_PARM_DESC(mcdi_logging_default,
		 "Enable MCDI logging on newly-probed functions");
#endif

static int efct_mcdi_rpc_async_internal(struct efct_nic *efct,
					struct efct_mcdi_cmd *cmd,
				       u32 *handle,
				       bool immediate_poll,
				       bool immediate_only);
static void efct_mcdi_start_or_queue(struct efct_mcdi_iface *mcdi,
				     bool allow_retry,
				    struct efct_mcdi_copy_buffer *copybuf,
				    struct list_head *cleanup_list);
static void efct_mcdi_cmd_start_or_queue(struct efct_mcdi_iface *mcdi,
					 struct efct_mcdi_cmd *cmd,
					struct efct_mcdi_copy_buffer *copybuf,
					struct list_head *cleanup_list);
static int efct_mcdi_cmd_start_or_queue_ext(struct efct_mcdi_iface *mcdi,
					    struct efct_mcdi_cmd *cmd,
					   struct efct_mcdi_copy_buffer *copybuf,
					   bool immediate_only,
					   struct list_head *cleanup_list);
static void efct_mcdi_poll_start(struct efct_mcdi_iface *mcdi,
				 struct efct_mcdi_cmd *cmd,
				struct efct_mcdi_copy_buffer *copybuf,
				struct list_head *cleanup_list);
static bool efct_mcdi_poll_once(struct efct_mcdi_iface *mcdi,
				struct efct_mcdi_cmd *cmd);
static bool efct_mcdi_complete_cmd(struct efct_mcdi_iface *mcdi,
				   struct efct_mcdi_cmd *cmd,
				  struct efct_mcdi_copy_buffer *copybuf,
				  struct list_head *cleanup_list);
static void efct_mcdi_timeout_cmd(struct efct_mcdi_iface *mcdi,
				  struct efct_mcdi_cmd *cmd,
				 struct list_head *cleanup_list);
static void efct_mcdi_reset_during_cmd(struct efct_mcdi_iface *mcdi,
				       struct efct_mcdi_cmd *cmd);
static void efct_mcdi_cmd_work(struct work_struct *work);
static void _efct_mcdi_mode_poll(struct efct_mcdi_iface *mcdi);
static void efct_mcdi_mode_fail(struct efct_nic *efct, struct list_head *cleanup_list);
static void _efct_mcdi_display_error(struct efct_nic *efct, u32 cmd,
				     size_t inlen, int raw, int arg, int rc);

static bool efct_cmd_running(struct efct_mcdi_cmd *cmd)
{
	return cmd->state == MCDI_STATE_RUNNING ||
	       cmd->state == MCDI_STATE_RUNNING_CANCELLED;
}

static bool efct_cmd_cancelled(struct efct_mcdi_cmd *cmd)
{
	return cmd->state == MCDI_STATE_RUNNING_CANCELLED ||
	       cmd->state == MCDI_STATE_PROXY_CANCELLED;
}

static void efct_mcdi_cmd_release(struct kref *ref)
{
	kfree(container_of(ref, struct efct_mcdi_cmd, ref));
}

static u32 efct_mcdi_cmd_handle(struct efct_mcdi_cmd *cmd)
{
	return cmd->handle;
}

static void _efct_mcdi_remove_cmd(struct efct_mcdi_iface *mcdi,
				  struct efct_mcdi_cmd *cmd,
				 struct list_head *cleanup_list)
{
	/* if cancelled, the completers have already been called */
	if (efct_cmd_cancelled(cmd))
		return;

	if (cmd->atomic_completer)
		cmd->atomic_completer(mcdi->efct, cmd->cookie, cmd->rc,
				      cmd->outbuf, cmd->outlen);
	if (cmd->completer) {
		list_add_tail(&cmd->cleanup_list, cleanup_list);
		++mcdi->outstanding_cleanups;
		kref_get(&cmd->ref);
	}
}

static void efct_mcdi_remove_cmd(struct efct_mcdi_iface *mcdi,
				 struct efct_mcdi_cmd *cmd,
				struct list_head *cleanup_list)
{
	list_del(&cmd->list);
	_efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);
	cmd->state = MCDI_STATE_FINISHED;
	kref_put(&cmd->ref, efct_mcdi_cmd_release);
	if (list_empty(&mcdi->cmd_list))
		wake_up(&mcdi->cmd_complete_wq);
}

static unsigned long efct_mcdi_rpc_timeout(struct efct_nic *efct, u32 cmd)
{
	if (!efct->type->mcdi_rpc_timeout)
		return MCDI_RPC_TIMEOUT;
	else
		return efct->type->mcdi_rpc_timeout(efct, cmd);
}

int efct_mcdi_init(struct efct_nic *efct)
{
	struct efct_mcdi_iface *mcdi;
	int rc = -ENOMEM;

	efct->mcdi = kzalloc(sizeof(*efct->mcdi), GFP_KERNEL);
	if (!efct->mcdi)
		goto fail;

	mcdi = efct_mcdi(efct);
	mcdi->efct = efct;

#ifdef CONFIG_XILINX_MCDI_LOGGING
	mcdi->logging_buffer = kmalloc(LOG_LINE_MAX, GFP_KERNEL);
	if (!mcdi->logging_buffer)
		goto fail2;
	mcdi->logging_enabled = mcdi_logging_default;
	efct->efct_dev->mcdi_logging = mcdi->logging_enabled;
#endif
	mcdi->workqueue = create_workqueue("mcdi_wq");
	if (!mcdi->workqueue)
		goto fail3;
	spin_lock_init(&mcdi->iface_lock);
	mcdi->mode = MCDI_MODE_POLL;
	INIT_LIST_HEAD(&mcdi->cmd_list);
	init_waitqueue_head(&mcdi->cmd_complete_wq);

	(void)efct_mcdi_poll_reboot(efct);
	mcdi->new_epoch = true;

	/* Recover from a failed assertion before probing */
	rc = efct_mcdi_handle_assertion(efct);
	if (rc) {
		netif_err(efct, probe, efct->net_dev,
			  "Unable to handle assertion\n");
		goto fail4;
	}

	/* Let the MC (and BMC, if this is a LOM) know that the driver
	 * is loaded. We should do this before we reset the NIC.
	 * This operation can specify the required firmware variant.
	 * For X3, the firmware Variant is 0
	 */
	rc = efct_mcdi_drv_attach(efct, MC_CMD_FW_FULL_FEATURED, &efct->mcdi->fn_flags, false);
	if (rc) {
		netif_err(efct, probe, efct->net_dev,
			  "Unable to register driver with MCPU\n");
		goto fail4;
	}

	return 0;
fail4:
	destroy_workqueue(mcdi->workqueue);
fail3:
#ifdef CONFIG_XILINX_MCDI_LOGGING
	kfree(mcdi->logging_buffer);
fail2:
#endif
	kfree(efct->mcdi);
	efct->mcdi = NULL;
fail:
	return rc;
}

void efct_mcdi_detach(struct efct_nic *efct)
{
	if (!efct->mcdi)
		return;

	if (!efct_nic_hw_unavailable(efct))
		/* Relinquish the device (back to the BMC, if this is a LOM) */
		efct_mcdi_drv_detach(efct);
}

void efct_mcdi_fini(struct efct_nic *efct)
{
	struct efct_mcdi_iface *iface;

	if (!efct->mcdi)
		return;

	efct_mcdi_wait_for_cleanup(efct);

	iface = efct_mcdi(efct);
#ifdef CONFIG_XILINX_MCDI_LOGGING
	kfree(iface->logging_buffer);
#endif

	destroy_workqueue(iface->workqueue);
	kfree(efct->mcdi);
	efct->mcdi = NULL;
}

static bool efct_mcdi_reset_cmd_running(struct efct_mcdi_iface *mcdi)
{
	struct efct_mcdi_cmd *cmd;

	list_for_each_entry(cmd, &mcdi->cmd_list, list)
		if (cmd->cmd == MC_CMD_REBOOT &&
		    efct_cmd_running(cmd))
			return true;
	return false;
}

static void efct_mcdi_reboot_detected(struct efct_nic *efct)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	struct efct_mcdi_cmd *cmd;

	if (!mcdi)
		return;

	_efct_mcdi_mode_poll(mcdi);
	list_for_each_entry(cmd, &mcdi->cmd_list, list)
		if (efct_cmd_running(cmd))
			cmd->reboot_seen = true;

	if (efct->type->mcdi_reboot_detected)
		efct->type->mcdi_reboot_detected(efct);
}

static bool efct_mcdi_wait_for_reboot(struct efct_nic *efct)
{
	size_t count;

	for (count = 0; count < MCDI_STATUS_DELAY_COUNT; ++count) {
		if (efct_mcdi_poll_reboot(efct)) {
			efct_mcdi_reboot_detected(efct);
			return true;
		}
		udelay(MCDI_STATUS_DELAY_US);
	}

	return false;
}

static bool efct_mcdi_flushed(struct efct_mcdi_iface *mcdi, bool ignore_cleanups)
{
	bool flushed;

	spin_lock_bh(&mcdi->iface_lock);
	flushed = list_empty(&mcdi->cmd_list) &&
		  (ignore_cleanups || !mcdi->outstanding_cleanups);
	spin_unlock_bh(&mcdi->iface_lock);
	return flushed;
}

/* Wait for outstanding MCDI commands to complete. */
void efct_mcdi_wait_for_cleanup(struct efct_nic *efct)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);

	if (!mcdi)
		return;

	wait_event(mcdi->cmd_complete_wq,
		   efct_mcdi_flushed(mcdi, false));
}

static void efct_mcdi_send_request(struct efct_nic *efct,
				   struct efct_mcdi_cmd *cmd)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	const efct_dword_t *inbuf = cmd->inbuf;
	size_t inlen = cmd->inlen;
	efct_dword_t hdr[2];
	size_t hdr_len;
	u32 xflags;
#ifdef CONFIG_XILINX_MCDI_LOGGING
	char *buf;
#endif

	if (!mcdi) {
		netif_err(efct, drv, efct->net_dev, "Cuaght null pointer for mcdi\n");
		return;
	}
#ifdef CONFIG_XILINX_MCDI_LOGGING
	buf = mcdi->logging_buffer; /* page-sized */
#endif

	mcdi->prev_seq = cmd->seq;
	mcdi->seq_held_by[cmd->seq] = cmd;
	mcdi->db_held_by = cmd;
	cmd->started = jiffies;

	xflags = 0;
	if (mcdi->mode == MCDI_MODE_EVENTS)
		xflags |= MCDI_HEADER_XFLAGS_EVREQ;

	if (efct->type->mcdi_max_ver == 1) {
		/* MCDI v1 */
		EFCT_POPULATE_DWORD_7(hdr[0],
				      MCDI_HEADER_RESPONSE, 0,
				     MCDI_HEADER_RESYNC, 1,
				     MCDI_HEADER_CODE, cmd->cmd,
				     MCDI_HEADER_DATALEN, inlen,
				     MCDI_HEADER_SEQ, cmd->seq,
				     MCDI_HEADER_XFLAGS, xflags,
				     MCDI_HEADER_NOT_EPOCH, !mcdi->new_epoch);
		hdr_len = 4;
	} else {
		/* MCDI v2 */
		WARN_ON(inlen > MCDI_CTL_SDU_LEN_MAX_V2);
		EFCT_POPULATE_DWORD_7(hdr[0],
				      MCDI_HEADER_RESPONSE, 0,
				     MCDI_HEADER_RESYNC, 1,
				     MCDI_HEADER_CODE, MC_CMD_V2_EXTN,
				     MCDI_HEADER_DATALEN, 0,
				     MCDI_HEADER_SEQ, cmd->seq,
				     MCDI_HEADER_XFLAGS, xflags,
				     MCDI_HEADER_NOT_EPOCH, !mcdi->new_epoch);
		EFCT_POPULATE_DWORD_2(hdr[1],
				      MC_CMD_V2_EXTN_IN_EXTENDED_CMD, cmd->cmd,
				     MC_CMD_V2_EXTN_IN_ACTUAL_LEN, inlen);
		hdr_len = 8;
	}

#ifdef CONFIG_XILINX_MCDI_LOGGING
	if (mcdi->logging_enabled && !WARN_ON_ONCE(!buf)) {
		const efct_dword_t *frags[] = { hdr, inbuf };
		size_t frag_len[] = { hdr_len, round_up(inlen, 4) };
		const efct_dword_t *frag;
		int bytes = 0;
		int i, j;
		u32 dcount = 0;
		/* Header length should always be a whole number of dwords,
		 * so scream if it's not.
		 */
		WARN_ON_ONCE(hdr_len % 4);

		for (j = 0; j < ARRAY_SIZE(frags); j++) {
			frag = frags[j];
			for (i = 0;
			     i < frag_len[j] / 4;
			     i++) {
				/* Do not exceed the internal printk limit.
				 * The string before that is just over 70 bytes.
				 */
				if ((bytes + 75) > LOG_LINE_MAX) {
					netif_info(efct, hw, efct->net_dev,
						   "MCDI RPC REQ:%s \\\n", buf);
					dcount = 0;
					bytes = 0;
				}
				bytes += snprintf(buf + bytes,
						  LOG_LINE_MAX - bytes, " %08x",
						  le32_to_cpu(frag[i].u32[0]));
				dcount++;
			}
		}

		netif_info(efct, hw, efct->net_dev, "MCDI RPC REQ:%s\n", buf);
	}
#endif

	efct->type->mcdi_request(efct, cmd->bufid, hdr, hdr_len, inbuf, inlen);

	mcdi->new_epoch = false;
}

static int efct_mcdi_errno(struct efct_nic *efct, u32 mcdi_err)
{
	switch (mcdi_err) {
	case 0:
	case MC_CMD_ERR_PROXY_PENDING:
	case MC_CMD_ERR_QUEUE_FULL:
		return mcdi_err;
	case MC_CMD_ERR_EPERM:
		return -EPERM;
	case MC_CMD_ERR_ENOENT:
		return -ENOENT;
	case MC_CMD_ERR_EINTR:
		return -EINTR;
	case MC_CMD_ERR_EAGAIN:
		return -EAGAIN;
	case MC_CMD_ERR_EACCES:
		return -EACCES;
	case MC_CMD_ERR_EBUSY:
		return -EBUSY;
	case MC_CMD_ERR_EINVAL:
		return -EINVAL;
	case MC_CMD_ERR_ERANGE:
		return -ERANGE;
	case MC_CMD_ERR_EDEADLK:
		return -EDEADLK;
	case MC_CMD_ERR_ENOSYS:
		return -ENOSYS;
	case MC_CMD_ERR_ETIME:
		return -ETIME;
	case MC_CMD_ERR_EALREADY:
		return -EALREADY;
	case MC_CMD_ERR_ENOSPC:
		return -ENOSPC;
	case MC_CMD_ERR_ENOMEM:
		return -ENOMEM;
	case MC_CMD_ERR_ENOTSUP:
		return -EOPNOTSUPP;
	case MC_CMD_ERR_ALLOC_FAIL:
		return -ENOBUFS;
	case MC_CMD_ERR_MAC_EXIST:
		return -EADDRINUSE;
	case MC_CMD_ERR_NO_EVB_PORT:
		return -EAGAIN;
	default:
		return -EPROTO;
	}
}

/* Test and clear MC-rebooted flag for this port/function; reset
 * software state as necessary.
 */
int efct_mcdi_poll_reboot(struct efct_nic *efct)
{
	if (!efct->mcdi)
		return 0;

	return efct->type->mcdi_poll_reboot(efct);
}

static void efct_mcdi_process_cleanup_list(struct efct_nic *efct,
					   struct list_head *cleanup_list)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	u32 cleanups = 0;

	while (!list_empty(cleanup_list)) {
		struct efct_mcdi_cmd *cmd =
			list_first_entry(cleanup_list,
					 struct efct_mcdi_cmd, cleanup_list);
		cmd->completer(efct, cmd->cookie, cmd->rc,
			       cmd->outbuf, cmd->outlen);
		list_del(&cmd->cleanup_list);
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
		++cleanups;
	}

	if (cleanups) {
		bool all_done;

		spin_lock_bh(&mcdi->iface_lock);
		EFCT_WARN_ON_PARANOID(cleanups > mcdi->outstanding_cleanups);
		all_done = (mcdi->outstanding_cleanups -= cleanups) == 0;
		spin_unlock_bh(&mcdi->iface_lock);
		if (all_done)
			wake_up(&mcdi->cmd_complete_wq);
	}
}

static void _efct_mcdi_cancel_cmd(struct efct_mcdi_iface *mcdi,
				  u32 handle,
				  struct list_head *cleanup_list)
{
	struct efct_nic *efct = mcdi->efct;
	struct efct_mcdi_cmd *cmd;

	list_for_each_entry(cmd, &mcdi->cmd_list, list)
		if (efct_mcdi_cmd_handle(cmd) == handle) {
			switch (cmd->state) {
			case MCDI_STATE_QUEUED:
			case MCDI_STATE_RETRY:
				netif_dbg(efct, drv, efct->net_dev,
					  "command %#x inlen %zu cancelled in queue\n",
					  cmd->cmd, cmd->inlen);
				/* if not yet running, properly cancel it */
				cmd->rc = -EPIPE;
				efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);
				break;
			case MCDI_STATE_RUNNING:
			case MCDI_STATE_PROXY:
				netif_dbg(efct, drv, efct->net_dev,
					  "command %#x inlen %zu cancelled after sending\n",
					  cmd->cmd, cmd->inlen);
				/* It's running. We can't cancel it on the MC,
				 * so we need to keep track of it so we can
				 * handle the response. We *also* need to call
				 * the command's completion function, and make
				 * sure it's not called again later, by
				 * marking it as cancelled.
				 */
				cmd->rc = -EPIPE;
				_efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);
				cmd->state = cmd->state == MCDI_STATE_RUNNING ?
					     MCDI_STATE_RUNNING_CANCELLED :
					     MCDI_STATE_PROXY_CANCELLED;
				break;
			case MCDI_STATE_RUNNING_CANCELLED:
			case MCDI_STATE_PROXY_CANCELLED:
				netif_warn(efct, drv, efct->net_dev,
					   "command %#x inlen %zu double cancelled\n",
					   cmd->cmd, cmd->inlen);
				break;
			case MCDI_STATE_FINISHED:
			default:
				/* invalid state? */
				WARN_ON(1);
			}
			break;
		}
}

void efct_mcdi_cancel_cmd(struct efct_nic *efct, u32 handle)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	LIST_HEAD(cleanup_list);

	if (!mcdi)
		return;

	spin_lock_bh(&mcdi->iface_lock);
	_efct_mcdi_cancel_cmd(mcdi, handle, &cleanup_list);
	spin_unlock_bh(&mcdi->iface_lock);
	efct_mcdi_process_cleanup_list(efct, &cleanup_list);
}

static void efct_mcdi_proxy_response(struct efct_mcdi_iface *mcdi,
				     struct efct_mcdi_cmd *cmd,
				    int status,
				    struct list_head *cleanup_list)
{
	mcdi->db_held_by = NULL;

	if (status) {
		/* status != 0 means don't retry */
		if (status == -EIO || status == -EINTR)
			efct_mcdi_reset_during_cmd(mcdi, cmd);
		kref_get(&cmd->ref);
		cmd->rc = status;
		efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);
		if (cancel_delayed_work(&cmd->work))
			kref_put(&cmd->ref, efct_mcdi_cmd_release);
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
	} else {
		/* status = 0 means ok to retry */
		efct_mcdi_cmd_start_or_queue(mcdi, cmd, NULL, cleanup_list);
	}
}

static void efct_mcdi_ev_proxy_response(struct efct_nic *efct,
					u32 handle, int status)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	struct efct_mcdi_cmd *cmd;
	LIST_HEAD(cleanup_list);
	bool found = false;

	spin_lock_bh(&mcdi->iface_lock);
	list_for_each_entry(cmd, &mcdi->cmd_list, list)
		if (cmd->state == MCDI_STATE_PROXY &&
		    cmd->proxy_handle == handle) {
			efct_mcdi_proxy_response(mcdi, cmd, status, &cleanup_list);
			found = true;
			break;
		}
	spin_unlock_bh(&mcdi->iface_lock);

	efct_mcdi_process_cleanup_list(efct, &cleanup_list);

	if (!found) {
		netif_err(efct, drv, efct->net_dev,
			  "MCDI proxy unexpected handle %#x\n",
			  handle);
		efct_schedule_reset(efct, RESET_TYPE_WORLD);
	}
}

static void efct_mcdi_ev_cpl(struct efct_nic *efct, u32 seqno,
			     u32 datalen, u32 mcdi_err)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	struct efct_mcdi_cmd *cmd;
	LIST_HEAD(cleanup_list);
	struct efct_mcdi_copy_buffer *copybuf;

	copybuf = kmalloc(sizeof(*copybuf), GFP_ATOMIC);
	if (!mcdi) {
		kfree(copybuf);
		return;
	}

	spin_lock(&mcdi->iface_lock);
	cmd = mcdi->seq_held_by[seqno];
	if (cmd) {
		kref_get(&cmd->ref);
		if (efct_mcdi_complete_cmd(mcdi, cmd, copybuf, &cleanup_list))
			if (cancel_delayed_work(&cmd->work))
				kref_put(&cmd->ref, efct_mcdi_cmd_release);
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
	} else {
		netif_err(efct, hw, efct->net_dev,
			  "MC response unexpected tx seq 0x%x\n",
			  seqno);
		/* this could theoretically just be a race between command
		 * time out and processing the completion event,  so while not
		 * a good sign, it'd be premature to attempt any recovery.
		 */
	}
	spin_unlock(&mcdi->iface_lock);

	efct_mcdi_process_cleanup_list(efct, &cleanup_list);

	kfree(copybuf);
}

static int
efct_mcdi_check_supported(struct efct_nic *efct, u32 cmd, size_t inlen)
{
	if (efct->type->mcdi_max_ver < 0 ||
	    (efct->type->mcdi_max_ver < 2 &&
	     cmd > MC_CMD_CMD_SPACE_ESCAPE_7)) {
		netif_err(efct, hw, efct->net_dev, "Invalid MCDI version\n");
		return -EINVAL;
	}

	if (inlen > MCDI_CTL_SDU_LEN_MAX_V2 ||
	    (efct->type->mcdi_max_ver < 2 &&
	     inlen > MCDI_CTL_SDU_LEN_MAX_V1)) {
		netif_err(efct, hw, efct->net_dev, "Invalid MCDI inlen\n");
		return -EMSGSIZE;
	}

	return 0;
}

struct efct_mcdi_blocking_data {
	struct kref ref;
	bool done;
	wait_queue_head_t wq;
	int rc;
	efct_dword_t *outbuf;
	size_t outlen;
	size_t outlen_actual;
};

#ifndef CONFIG_EFCT_TEST
static void efct_mcdi_blocking_data_release(struct kref *ref)
{
	kfree(container_of(ref, struct efct_mcdi_blocking_data, ref));
}

static void efct_mcdi_rpc_completer(struct efct_nic *efct, unsigned long cookie,
				    int rc, efct_dword_t *outbuf,
				   size_t outlen_actual)
{
	struct efct_mcdi_blocking_data *wait_data =
		(struct efct_mcdi_blocking_data *)cookie;

	wait_data->rc = rc;
	memcpy(wait_data->outbuf, outbuf,
	       min(outlen_actual, wait_data->outlen));
	wait_data->outlen_actual = outlen_actual;
	/*memory barrier*/
	smp_wmb();
	wait_data->done = true;
	wake_up(&wait_data->wq);
	kref_put(&wait_data->ref, efct_mcdi_blocking_data_release);
}

static int efct_mcdi_rpc_sync(struct efct_nic *efct, u32 cmd,
			      const efct_dword_t *inbuf, size_t inlen,
			      efct_dword_t *outbuf, size_t outlen,
			      size_t *outlen_actual, bool quiet)
{
	struct efct_mcdi_blocking_data *wait_data;
	struct efct_mcdi_cmd *cmd_item;
	u32 handle;
	int rc;

	if (outlen_actual)
		*outlen_actual = 0;

	wait_data = kmalloc(sizeof(*wait_data), GFP_KERNEL);
	if (!wait_data)
		return -ENOMEM;

	cmd_item = kmalloc(sizeof(*cmd_item), GFP_KERNEL);
	if (!cmd_item) {
		kfree(wait_data);
		return -ENOMEM;
	}

	kref_init(&wait_data->ref);
	wait_data->done = false;
	init_waitqueue_head(&wait_data->wq);
	wait_data->outbuf = outbuf;
	wait_data->outlen = outlen;

	kref_init(&cmd_item->ref);
	cmd_item->quiet = quiet;
	cmd_item->cookie = (unsigned long)wait_data;
	cmd_item->atomic_completer = NULL;
	cmd_item->completer = &efct_mcdi_rpc_completer;
	cmd_item->cmd = cmd;
	cmd_item->inlen = inlen;
	cmd_item->inbuf = inbuf;

	/* Claim an extra reference for the completer to put. */
	kref_get(&wait_data->ref);
	rc = efct_mcdi_rpc_async_internal(efct, cmd_item, &handle, true, false);
	if (rc) {
		kref_put(&wait_data->ref, efct_mcdi_blocking_data_release);
		goto out;
	}

	if (!wait_event_timeout(wait_data->wq, wait_data->done,
				MCDI_ACQUIRE_TIMEOUT +
				efct_mcdi_rpc_timeout(efct, cmd)) &&
	    !wait_data->done) {
		netif_err(efct, drv, efct->net_dev,
			  "MC command 0x%x inlen %zu timed out (sync)\n",
			  cmd, inlen);

		efct_mcdi_cancel_cmd(efct, handle);

		wait_data->rc = -ETIMEDOUT;
		wait_data->outlen_actual = 0;
	}

	if (outlen_actual)
		*outlen_actual = wait_data->outlen_actual;
	rc = wait_data->rc;

out:
	kref_put(&wait_data->ref, efct_mcdi_blocking_data_release);

	return rc;
}
#endif

int efct_mcdi_rpc_async_ext(struct efct_nic *efct, u32 cmd,
			    const efct_dword_t *inbuf, size_t inlen,
			   efct_mcdi_async_completer *atomic_completer,
			   efct_mcdi_async_completer *completer,
			   unsigned long cookie, bool quiet,
			   bool immediate_only, u32 *handle)
{
	struct efct_mcdi_cmd *cmd_item =
		kmalloc(sizeof(struct efct_mcdi_cmd) + inlen, GFP_ATOMIC);

	if (!cmd_item)
		return -ENOMEM;

	kref_init(&cmd_item->ref);
	cmd_item->quiet = quiet;
	cmd_item->cookie = cookie;
	cmd_item->completer = completer;
	cmd_item->atomic_completer = atomic_completer;
	cmd_item->cmd = cmd;
	cmd_item->inlen = inlen;
	/* inbuf is probably not valid after return, so take a copy */
	cmd_item->inbuf = (efct_dword_t *)(cmd_item + 1);
	memcpy(cmd_item + 1, inbuf, inlen);

	return efct_mcdi_rpc_async_internal(efct, cmd_item, handle, false,
					   immediate_only);
}

static bool efct_mcdi_get_seq(struct efct_mcdi_iface *mcdi, unsigned char *seq)
{
	*seq = mcdi->prev_seq;
	do {
		*seq = (*seq + 1) % ARRAY_SIZE(mcdi->seq_held_by);
	} while (mcdi->seq_held_by[*seq] && *seq != mcdi->prev_seq);
	return !mcdi->seq_held_by[*seq];
}

static int efct_mcdi_rpc_async_internal(struct efct_nic *efct,
					struct efct_mcdi_cmd *cmd,
				       u32 *handle,
				       bool immediate_poll, bool immediate_only)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	struct efct_mcdi_copy_buffer *copybuf;
	LIST_HEAD(cleanup_list);
	int rc;

	rc = efct_mcdi_check_supported(efct, cmd->cmd, cmd->inlen);
	if (rc) {
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
		return rc;
	}
	if (!mcdi || efct->mc_bist_for_other_fn) {
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
		return -ENETDOWN;
	}

	copybuf = immediate_poll ?
		  kmalloc(sizeof(struct efct_mcdi_copy_buffer), GFP_KERNEL) :
		  NULL;

	cmd->mcdi = mcdi;
	INIT_DELAYED_WORK(&cmd->work, efct_mcdi_cmd_work);
	INIT_LIST_HEAD(&cmd->list);
	INIT_LIST_HEAD(&cmd->cleanup_list);
	cmd->proxy_handle = 0;
	cmd->rc = 0;
	cmd->outbuf = NULL;
	cmd->outlen = 0;

	spin_lock_bh(&mcdi->iface_lock);

	if (mcdi->mode == MCDI_MODE_FAIL) {
		spin_unlock_bh(&mcdi->iface_lock);
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
		kfree(copybuf);
		return -ENETDOWN;
	}

	cmd->handle = mcdi->prev_handle++;
	if (handle)
		*handle = efct_mcdi_cmd_handle(cmd);

	list_add_tail(&cmd->list, &mcdi->cmd_list);
	rc = efct_mcdi_cmd_start_or_queue_ext(mcdi, cmd, copybuf, immediate_only,
					      &cleanup_list);
	if (rc) {
		list_del(&cmd->list);
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
	}

	spin_unlock_bh(&mcdi->iface_lock);

	efct_mcdi_process_cleanup_list(efct, &cleanup_list);

	kfree(copybuf);

	return rc;
}

static int efct_mcdi_cmd_start_or_queue_ext(struct efct_mcdi_iface *mcdi,
					    struct efct_mcdi_cmd *cmd,
					   struct efct_mcdi_copy_buffer *copybuf,
					   bool immediate_only,
					   struct list_head *cleanup_list)
{
	struct efct_nic *efct = mcdi->efct;
	u8 seq, bufid;
	int rc = 0;

	if (!mcdi->db_held_by &&
	    efct_mcdi_get_seq(mcdi, &seq) &&
	    efct->type->mcdi_get_buf(efct, &bufid)) {
		cmd->seq = seq;
		cmd->bufid = bufid;
		cmd->polled = mcdi->mode == MCDI_MODE_POLL;
		cmd->reboot_seen = false;
		efct_mcdi_send_request(efct, cmd);
		cmd->state = MCDI_STATE_RUNNING;

		if (cmd->polled) {
			efct_mcdi_poll_start(mcdi, cmd, copybuf, cleanup_list);
		} else {
			kref_get(&cmd->ref);
			queue_delayed_work(mcdi->workqueue, &cmd->work,
					   efct_mcdi_rpc_timeout(efct, cmd->cmd));
		}
	} else if (immediate_only) {
		rc = -EAGAIN;
	} else {
		cmd->state = MCDI_STATE_QUEUED;
	}

	return rc;
}

static void efct_mcdi_cmd_start_or_queue(struct efct_mcdi_iface *mcdi,
					 struct efct_mcdi_cmd *cmd,
					struct efct_mcdi_copy_buffer *copybuf,
					struct list_head *cleanup_list)
{
	/* when immediate_only=false this can only return success */
	(void)efct_mcdi_cmd_start_or_queue_ext(mcdi, cmd, copybuf, false,
					       cleanup_list);
}

/* try to advance other commands */
static void efct_mcdi_start_or_queue(struct efct_mcdi_iface *mcdi,
				     bool allow_retry,
				    struct efct_mcdi_copy_buffer *copybuf,
				    struct list_head *cleanup_list)
{
	struct efct_mcdi_cmd *cmd, *tmp;

	list_for_each_entry_safe(cmd, tmp, &mcdi->cmd_list, list)
		if (cmd->state == MCDI_STATE_QUEUED ||
		    (cmd->state == MCDI_STATE_RETRY && allow_retry))
			efct_mcdi_cmd_start_or_queue(mcdi, cmd, copybuf,
						     cleanup_list);
}

static void efct_mcdi_poll_start(struct efct_mcdi_iface *mcdi,
				 struct efct_mcdi_cmd *cmd,
				struct efct_mcdi_copy_buffer *copybuf,
				struct list_head *cleanup_list)
{
	/* Poll for completion. Poll quickly (once a us) for the 1st jiffy,
	 * because generally mcdi responses are fast. After that, back off
	 * and poll once a jiffy (approximately)
	 */
	int spins = copybuf ? USER_TICK_USEC : 0;

	while (spins) {
		if (efct_mcdi_poll_once(mcdi, cmd)) {
			efct_mcdi_complete_cmd(mcdi, cmd, copybuf, cleanup_list);
			return;
		}

		--spins;
		udelay(1);
	}

	/* didn't get a response in the first jiffy;
	 * schedule poll after another jiffy
	 */
	kref_get(&cmd->ref);
	queue_delayed_work(mcdi->workqueue, &cmd->work, 1);
}

static bool efct_mcdi_poll_once(struct efct_mcdi_iface *mcdi,
				struct efct_mcdi_cmd *cmd)
{
	struct efct_nic *efct = mcdi->efct;

	/* complete or error, either way return true */
	return efct_nic_hw_unavailable(efct) ||
	       efct->type->mcdi_poll_response(efct, cmd->bufid);
}

static unsigned long efct_mcdi_poll_interval(struct efct_mcdi_iface *mcdi,
					     struct efct_mcdi_cmd *cmd)
{
	if (time_before(jiffies, cmd->started + msecs_to_jiffies(10)))
		return msecs_to_jiffies(1);
	else if (time_before(jiffies, cmd->started + msecs_to_jiffies(100)))
		return msecs_to_jiffies(10);
	else if (time_before(jiffies, cmd->started + msecs_to_jiffies(1000)))
		return msecs_to_jiffies(100);
	else
		return msecs_to_jiffies(1000);
}

static bool efct_mcdi_check_timeout(struct efct_mcdi_iface *mcdi,
				    struct efct_mcdi_cmd *cmd)
{
	return time_after(jiffies, cmd->started +
				   efct_mcdi_rpc_timeout(mcdi->efct, cmd->cmd));
}

static void efct_mcdi_proxy_timeout_cmd(struct efct_mcdi_iface *mcdi,
					struct efct_mcdi_cmd *cmd,
				       struct list_head *cleanup_list)
{
	struct efct_nic *efct = mcdi->efct;

	netif_err(efct, drv, efct->net_dev, "MCDI proxy timeout (handle %#x)\n",
		  cmd->proxy_handle);

	cmd->rc = -ETIMEDOUT;
	efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);

	efct_mcdi_mode_fail(efct, cleanup_list);
	efct_schedule_reset(efct, RESET_TYPE_MCDI_TIMEOUT);
}

static void efct_mcdi_cmd_work(struct work_struct *context)
{
	struct efct_mcdi_cmd *cmd =
#if !defined(EFCT_USE_KCOMPAT) || !defined(EFCT_NEED_WORK_API_WRAPPERS)
		container_of(context, struct efct_mcdi_cmd, work.work);
#else
		container_of(context, struct efct_mcdi_cmd, work);
#endif
	struct efct_mcdi_iface *mcdi = cmd->mcdi;
	struct efct_mcdi_copy_buffer *copybuf =
		kmalloc(sizeof(struct efct_mcdi_copy_buffer), GFP_KERNEL);
	LIST_HEAD(cleanup_list);

	spin_lock_bh(&mcdi->iface_lock);

	if (cmd->state == MCDI_STATE_FINISHED) {
		/* The command is done and this is a race between the
		 * completion in another thread and the work item running.
		 * All processing been done, so just release it.
		 */
		spin_unlock_bh(&mcdi->iface_lock);
		kref_put(&cmd->ref, efct_mcdi_cmd_release);
		kfree(copybuf);
		return;
	}

	EFCT_WARN_ON_PARANOID(cmd->state == MCDI_STATE_QUEUED);
	EFCT_WARN_ON_PARANOID(cmd->state == MCDI_STATE_RETRY);

	/* if state PROXY, then proxy time out */
	if (cmd->state == MCDI_STATE_PROXY) {
		efct_mcdi_proxy_timeout_cmd(mcdi, cmd, &cleanup_list);
	/* else running, check for completion */
	} else if (efct_mcdi_poll_once(mcdi, cmd)) {
		if (!cmd->polled) {
			/* check whether the event is pending on EVQ0 */
			if (efct_nic_mcdi_ev_pending(mcdi->efct, 0))
				netif_err(mcdi->efct, drv, mcdi->efct->net_dev,
					  "MC command 0x%x inlen %zu mode %d completed without an interrupt after %u ms\n",
					  cmd->cmd, cmd->inlen,
					  cmd->polled ? MCDI_MODE_POLL : MCDI_MODE_EVENTS,
					  jiffies_to_msecs(jiffies - cmd->started));
			else
				netif_err(mcdi->efct, drv, mcdi->efct->net_dev,
					  "MC command 0x%x inlen %zu mode %d completed without an event after %u ms\n",
					  cmd->cmd, cmd->inlen,
					  cmd->polled ? MCDI_MODE_POLL : MCDI_MODE_EVENTS,
					  jiffies_to_msecs(jiffies - cmd->started));
		}
		efct_mcdi_complete_cmd(mcdi, cmd, copybuf, &cleanup_list);
	/* then check for timeout. If evented, it must have timed out */
	} else if (!cmd->polled || efct_mcdi_check_timeout(mcdi, cmd)) {
		if (efct_mcdi_wait_for_reboot(mcdi->efct)) {
			netif_err(mcdi->efct, drv, mcdi->efct->net_dev,
				  "MC command 0x%x inlen %zu state %d mode %d timed out after %u ms during mc reboot\n",
				cmd->cmd, cmd->inlen, cmd->state,
				cmd->polled ? MCDI_MODE_POLL : MCDI_MODE_EVENTS,
				jiffies_to_msecs(jiffies - cmd->started));
			efct_mcdi_complete_cmd(mcdi, cmd, copybuf, &cleanup_list);
		} else {
			efct_mcdi_timeout_cmd(mcdi, cmd, &cleanup_list);
		}

	/* else reschedule for another poll */
	} else {
		kref_get(&cmd->ref);
		queue_delayed_work(mcdi->workqueue, &cmd->work,
				   efct_mcdi_poll_interval(mcdi, cmd));
	}

	spin_unlock_bh(&mcdi->iface_lock);

	kref_put(&cmd->ref, efct_mcdi_cmd_release);

	efct_mcdi_process_cleanup_list(mcdi->efct, &cleanup_list);

	kfree(copybuf);
}

static void efct_mcdi_reset_during_cmd(struct efct_mcdi_iface *mcdi,
				       struct efct_mcdi_cmd *cmd)
{
	bool reset_running = efct_mcdi_reset_cmd_running(mcdi);
	struct efct_nic *efct = mcdi->efct;
	int rc;

	if (!reset_running)
		netif_err(efct, hw, efct->net_dev,
			  "Command %#x inlen %zu cancelled by MC reboot\n",
			  cmd->cmd, cmd->inlen);
	rc = efct_mcdi_wait_for_reboot(efct);
	/* consume the reset notification if we haven't already */
	if (!cmd->reboot_seen && rc)
		if (!reset_running)
			efct_schedule_reset(efct, RESET_TYPE_MC_FAILURE);
}

/* Returns true if the MCDI module is finished with the command.
 * (examples of false would be if the command was proxied, or it was
 * rejected by the MC due to lack of resources and requeued).
 */
static bool efct_mcdi_complete_cmd(struct efct_mcdi_iface *mcdi,
				   struct efct_mcdi_cmd *cmd,
				  struct efct_mcdi_copy_buffer *copybuf,
				  struct list_head *cleanup_list)
{
	struct efct_nic *efct = mcdi->efct;
	size_t resp_hdr_len, resp_data_len;
	bool completed = false;
	u8 bufid = cmd->bufid;
	efct_dword_t *outbuf;
	u32 respcmd, error;
	efct_dword_t hdr;
	int rc;

	outbuf = copybuf ? copybuf->buffer : NULL;
	/* ensure the command can't go away before this function returns */
	kref_get(&cmd->ref);

	efct->type->mcdi_read_response(efct, bufid, &hdr, 0, 4);
	respcmd = EFCT_DWORD_FIELD(hdr, MCDI_HEADER_CODE);
	error = EFCT_DWORD_FIELD(hdr, MCDI_HEADER_ERROR);

	if (respcmd != MC_CMD_V2_EXTN) {
		resp_hdr_len = 4;
		resp_data_len = EFCT_DWORD_FIELD(hdr, MCDI_HEADER_DATALEN);
	} else {
		efct->type->mcdi_read_response(efct, bufid, &hdr, 4, 4);
		respcmd = EFCT_DWORD_FIELD(hdr, MC_CMD_V2_EXTN_IN_EXTENDED_CMD);
		resp_hdr_len = 8;
		resp_data_len =
			EFCT_DWORD_FIELD(hdr, MC_CMD_V2_EXTN_IN_ACTUAL_LEN);
	}

#ifdef CONFIG_XILINX_MCDI_LOGGING
	if (mcdi->logging_enabled && !WARN_ON_ONCE(!mcdi->logging_buffer)) {
		size_t len;
		int bytes = 0;
		int i;
		u32 dcount = 0;
		char *log = mcdi->logging_buffer;

		WARN_ON_ONCE(resp_hdr_len % 4);
		/* MCDI_DECLARE_BUF ensures that underlying buffer is padded
		 * to dword size, and the MCDI buffer is always dword size
		 */
		len = resp_hdr_len / 4 + DIV_ROUND_UP(resp_data_len, 4);

		for (i = 0; i < len; i++) {
			if ((bytes + 75) > LOG_LINE_MAX) {
				netif_info(efct, hw, efct->net_dev,
					   "MCDI RPC RESP:%s \\\n", log);
				dcount = 0;
				bytes = 0;
			}
			efct->type->mcdi_read_response(efct, bufid,
						      &hdr, (i * 4), 4);
			bytes += snprintf(log + bytes, LOG_LINE_MAX - bytes,
					" %08x", le32_to_cpu(hdr.u32[0]));
			dcount++;
		}

		netif_info(efct, hw, efct->net_dev, "MCDI RPC RESP:%s\n", log);
	}
#endif

	if (error && resp_data_len == 0) {
		/* MC rebooted during command */
		efct_mcdi_reset_during_cmd(mcdi, cmd);
		rc = -EIO;
	} else if (!outbuf) {
		rc = -ENOMEM;
	} else {
		if (WARN_ON_ONCE(error && resp_data_len < 4))
			resp_data_len = 4;

		efct->type->mcdi_read_response(efct, bufid, outbuf,
					      resp_hdr_len, resp_data_len);

		if (error) {
			rc = EFCT_DWORD_FIELD(outbuf[0], EFCT_DWORD_0);
			if (!cmd->quiet) {
				int err_arg = 0;

#ifdef WITH_MCDI_V2
				if (resp_data_len >= MC_CMD_ERR_ARG_OFST + 4) {
					efct->type->mcdi_read_response(efct, bufid, &hdr,
								      resp_hdr_len +
								      MC_CMD_ERR_ARG_OFST, 4);
					err_arg = EFCT_DWORD_VAL(hdr);
				}
#endif
				_efct_mcdi_display_error(efct, cmd->cmd,
							 cmd->inlen, rc, err_arg,
							efct_mcdi_errno(efct, rc));
			}
			rc = efct_mcdi_errno(efct, rc);
		} else {
			rc = 0;
		}
	}

	if (rc == MC_CMD_ERR_PROXY_PENDING) {
		if (mcdi->db_held_by != cmd || cmd->proxy_handle ||
		    resp_data_len < MC_CMD_ERR_PROXY_PENDING_HANDLE_OFST + 4) {
			/* The MC shouldn't return the doorbell early and then
			 * proxy. It also shouldn't return PROXY_PENDING with
			 * no handle or proxy a command that's already been
			 * proxied. Schedule an flr to reset the state.
			 */
			if (mcdi->db_held_by != cmd)
				netif_err(efct, drv, efct->net_dev,
					  "MCDI proxy pending with early db return\n");
			if (cmd->proxy_handle)
				netif_err(efct, drv, efct->net_dev,
					  "MCDI proxy pending twice\n");
			if (resp_data_len <
			    MC_CMD_ERR_PROXY_PENDING_HANDLE_OFST + 4)
				netif_err(efct, drv, efct->net_dev,
					  "MCDI proxy pending with no handle\n");
			cmd->rc = -EIO;
			efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);
			completed = true;

			efct_mcdi_mode_fail(efct, cleanup_list);
			efct_schedule_reset(efct, RESET_TYPE_MCDI_TIMEOUT);
		} else {
			/* keep the doorbell. no commands
			 * can be issued until the proxy response.
			 */
			cmd->state = MCDI_STATE_PROXY;
			efct->type->mcdi_read_response(efct, bufid, &hdr,
				resp_hdr_len +
					MC_CMD_ERR_PROXY_PENDING_HANDLE_OFST,
				4);
			cmd->proxy_handle = EFCT_DWORD_FIELD(hdr, EFCT_DWORD_0);
			kref_get(&cmd->ref);
			queue_delayed_work(mcdi->workqueue, &cmd->work,
					   MCDI_PROXY_TIMEOUT);
		}
	} else {
		/* free doorbell */
		if (mcdi->db_held_by == cmd)
			mcdi->db_held_by = NULL;

		if (efct_cmd_cancelled(cmd)) {
			list_del(&cmd->list);
			kref_put(&cmd->ref, efct_mcdi_cmd_release);
			completed = true;
		} else if (rc == MC_CMD_ERR_QUEUE_FULL) {
			cmd->state = MCDI_STATE_RETRY;
		} else {
			cmd->rc = rc;
			cmd->outbuf = outbuf;
			cmd->outlen = outbuf ? resp_data_len : 0;
			efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);
			completed = true;
		}
	}

	/* free sequence number and buffer */
	mcdi->seq_held_by[cmd->seq] = NULL;
	efct->type->mcdi_put_buf(efct, bufid);

	efct_mcdi_start_or_queue(mcdi, rc != MC_CMD_ERR_QUEUE_FULL,
				 NULL, cleanup_list);

	/* wake up anyone waiting for flush */
	wake_up(&mcdi->cmd_complete_wq);

	kref_put(&cmd->ref, efct_mcdi_cmd_release);

	return completed;
}

static void efct_mcdi_timeout_cmd(struct efct_mcdi_iface *mcdi,
				  struct efct_mcdi_cmd *cmd,
				 struct list_head *cleanup_list)
{
	struct efct_nic *efct = mcdi->efct;

	netif_err(efct, drv, efct->net_dev,
		  "MC command 0x%x inlen %zu state %d mode %d timed out after %u ms\n",
		  cmd->cmd, cmd->inlen, cmd->state,
		  cmd->polled ? MCDI_MODE_POLL : MCDI_MODE_EVENTS,
		  jiffies_to_msecs(jiffies - cmd->started));

	efct->type->mcdi_put_buf(efct, cmd->bufid);

	cmd->rc = -ETIMEDOUT;
	efct_mcdi_remove_cmd(mcdi, cmd, cleanup_list);

	efct_mcdi_mode_fail(efct, cleanup_list);
	efct_schedule_reset(efct, RESET_TYPE_MCDI_TIMEOUT);
}

/**
 * efct_mcdi_rpc - Issue an MCDI command and wait for completion
 * @efct: NIC through which to issue the command
 * @cmd: Command type number
 * @inbuf: Command parameters
 * @inlen: Length of command parameters, in bytes.  Must be a multiple
 *	of 4 and no greater than %MCDI_CTL_SDU_LEN_MAX_V1.
 * @outbuf: Response buffer.  May be %NULL if @outlen is 0.
 * @outlen: Length of response buffer, in bytes.  If the actual
 *	response is longer than @outlen & ~3, it will be truncated
 *	to that length.
 * @outlen_actual: Pointer through which to return the actual response
 *	length.  May be %NULL if this is not needed.
 *
 * This function may sleep and therefore must be called in process
 * context.
 *
 * Return: A negative error code, or zero if successful.  The error
 *	code may come from the MCDI response or may indicate a failure
 *	to communicate with the MC.  In the former case, the response
 *	will still be copied to @outbuf and *@outlen_actual will be
 *	set accordingly.  In the latter case, *@outlen_actual will be
 *	set to zero.
 */
int efct_mcdi_rpc(struct efct_nic *efct, u32 cmd,
		  const efct_dword_t *inbuf, size_t inlen,
		 efct_dword_t *outbuf, size_t outlen,
		 size_t *outlen_actual)
{
	return efct_mcdi_rpc_sync(efct, cmd, inbuf, inlen, outbuf, outlen,
				 outlen_actual, false);
}

/* Normally, on receiving an error code in the MCDI response,
 * efct_mcdi_rpc will log an error message containing (among other
 * things) the raw error code, by means of efct_mcdi_display_error.
 * This _quiet version suppresses that; if the caller wishes to log
 * the error conditionally on the return code, it should call this
 * function and is then responsible for calling efct_mcdi_display_error
 * as needed.
 */
int efct_mcdi_rpc_quiet(struct efct_nic *efct, u32 cmd,
			const efct_dword_t *inbuf, size_t inlen,
		       efct_dword_t *outbuf, size_t outlen,
		       size_t *outlen_actual)
{
	return efct_mcdi_rpc_sync(efct, cmd, inbuf, inlen, outbuf, outlen,
				 outlen_actual, true);
}

/**
 * efct_mcdi_rpc_async - Schedule an MCDI command to run asynchronously
 * @efct: NIC through which to issue the command
 * @cmd: Command type number
 * @inbuf: Command parameters
 * @inlen: Length of command parameters, in bytes
 * @complete: Function to be called on completion or cancellation.
 * @cookie: Arbitrary value to be passed to @complete.
 *
 * This function does not sleep and therefore may be called in atomic
 * context.  It will fail if event queues are disabled or if MCDI
 * event completions have been disabled due to an error.
 *
 * If it succeeds, the @complete function will be called exactly once
 * in atomic context, when one of the following occurs:
 * (a) the completion event is received (in NAPI context)
 * (b) event queues are disabled (in the process that disables them)
 * (c) the request times-out (in timer context)
 */
int
efct_mcdi_rpc_async(struct efct_nic *efct, u32 cmd,
		    const efct_dword_t *inbuf, size_t inlen,
		   efct_mcdi_async_completer *complete, unsigned long cookie)
{
	return efct_mcdi_rpc_async_ext(efct, cmd, inbuf, inlen, NULL,
				      complete, cookie, false, false, NULL);
}

int efct_mcdi_rpc_async_quiet(struct efct_nic *efct, u32 cmd,
			      const efct_dword_t *inbuf, size_t inlen,
			     efct_mcdi_async_completer *complete,
			     unsigned long cookie)
{
	return efct_mcdi_rpc_async_ext(efct, cmd, inbuf, inlen, NULL,
				      complete, cookie, true, false, NULL);
}

static void _efct_mcdi_display_error(struct efct_nic *efct, u32 cmd,
				     size_t inlen, int raw, int arg, int rc)
{
	if (efct->net_dev)
		netif_cond_dbg(efct, hw, efct->net_dev,
			       rc == -EPERM || efct_nic_hw_unavailable(efct), err,
			       "MC command 0x%x inlen %d failed rc=%d (raw=%d) arg=%d\n",
			       cmd, (int)inlen, rc, raw, arg);
	else
		pci_err(efct->efct_dev->pci_dev,
			"MC command 0x%x inlen %d failed rc=%d (raw=%d) arg=%d\n",
			cmd, (int)inlen, rc, raw, arg);
}

void efct_mcdi_display_error(struct efct_nic *efct, u32 cmd,
			     size_t inlen, efct_dword_t *outbuf,
			    size_t outlen, int rc)
{
	int code = 0, arg = 0;

	if (outlen >= MC_CMD_ERR_CODE_OFST + 4)
		code = MCDI_DWORD(outbuf, ERR_CODE);
#ifdef WITH_MCDI_V2
	if (outlen >= MC_CMD_ERR_ARG_OFST + 4)
		arg = MCDI_DWORD(outbuf, ERR_ARG);
#endif

	_efct_mcdi_display_error(efct, cmd, inlen, code, arg, rc);
}

/* Switch to polled MCDI completions. */
static void _efct_mcdi_mode_poll(struct efct_mcdi_iface *mcdi)
{
	/* If already in polling mode, nothing to do.
	 * If in fail-fast state, don't switch to polled completion, FLR
	 * recovery will do that later.
	 */
	if (mcdi->mode == MCDI_MODE_EVENTS) {
		struct efct_mcdi_cmd *cmd;

		mcdi->mode = MCDI_MODE_POLL;

		list_for_each_entry(cmd, &mcdi->cmd_list, list)
			if (efct_cmd_running(cmd) && !cmd->polled) {
				netif_dbg(mcdi->efct, drv, mcdi->efct->net_dev,
					  "converting command %#x inlen %zu to polled mode\n",
					  cmd->cmd, cmd->inlen);
				cmd->polled = true;
				if (cancel_delayed_work(&cmd->work))
					queue_delayed_work(mcdi->workqueue,
							   &cmd->work, 0);
			}
	}
}

/* Set MCDI mode to fail to prevent any new commands, then cancel any
 * outstanding commands.
 * Caller must hold the mcdi iface_lock.
 */
static void efct_mcdi_mode_fail(struct efct_nic *efct, struct list_head *cleanup_list)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	struct efct_mcdi_cmd *cmd;

	if (!mcdi)
		return;

	mcdi->mode = MCDI_MODE_FAIL;

	while (!list_empty(&mcdi->cmd_list)) {
		cmd = list_first_entry(&mcdi->cmd_list, struct efct_mcdi_cmd,
				       list);
		_efct_mcdi_cancel_cmd(mcdi, efct_mcdi_cmd_handle(cmd), cleanup_list);
	}
}

static void efct_mcdi_ev_death(struct efct_nic *efct, bool bist)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);

	if (bist) {
		efct->mc_bist_for_other_fn = true;
		efct->type->mcdi_record_bist_event(efct);
	}
	spin_lock(&mcdi->iface_lock);
	efct_mcdi_reboot_detected(efct);
	/* if this is the result of a MC_CMD_REBOOT then don't schedule reset */
	if (bist || !efct_mcdi_reset_cmd_running(mcdi))
		efct_schedule_reset(efct, bist ? RESET_TYPE_MC_BIST :
					       RESET_TYPE_MC_FAILURE);
	spin_unlock(&mcdi->iface_lock);
}

bool efct_mcdi_process_event(struct efct_nic *efct,
			     efct_qword_t *event)
{
	int code = EFCT_QWORD_FIELD(*event, MCDI_EVENT_CODE);
	u32 data = EFCT_QWORD_FIELD(*event, MCDI_EVENT_DATA);

	switch (code) {
	case MCDI_EVENT_CODE_BADSSERT:
		netif_err(efct, hw, efct->net_dev,
			  "MC watchdog or assertion failure at 0x%x\n", data);
		efct_mcdi_ev_death(efct, false);
		return true;

	case MCDI_EVENT_CODE_PMNOTICE:
		netif_info(efct, wol, efct->net_dev, "MCDI PM event.\n");
		return true;

	case MCDI_EVENT_CODE_CMDDONE:
		efct_mcdi_ev_cpl(efct,
				 MCDI_EVENT_FIELD(*event, CMDDONE_SEQ),
				MCDI_EVENT_FIELD(*event, CMDDONE_DATALEN),
				MCDI_EVENT_FIELD(*event, CMDDONE_ERRNO));
		return true;
	case MCDI_EVENT_CODE_PROXY_RESPONSE:
		efct_mcdi_ev_proxy_response(efct,
					    MCDI_EVENT_FIELD(*event, PROXY_RESPONSE_HANDLE),
					   MCDI_EVENT_FIELD(*event, PROXY_RESPONSE_RC));
		return true;
	case MCDI_EVENT_CODE_SCHEDERR:
		netif_dbg(efct, hw, efct->net_dev,
			  "MC Scheduler alert (0x%x)\n", data);
		return true;
	case MCDI_EVENT_CODE_REBOOT:
	case MCDI_EVENT_CODE_MC_REBOOT: /* XXX should handle this differently? */
		efct_mcdi_ev_death(efct, false);
		return true;
	case MCDI_EVENT_CODE_MC_BIST:
		netif_info(efct, hw, efct->net_dev, "MC entered BIST mode\n");
		efct_mcdi_ev_death(efct, true);
		return true;
#ifdef CONFIG_XILINX_PTP
	case MCDI_EVENT_CODE_PTP_PPS:
	case MCDI_EVENT_CODE_HW_PPS:
		efct_ptp_event(efct, event);
		return true;
#endif
	}

	return false;
}

/**************************************************************************
 *
 * Specific request functions
 *
 **************************************************************************
 */

static int efct_mcdi_drv_attach_attempt(struct efct_nic *efct,
					u32 fw_variant, u32 new_state,
				       u32 *flags, bool reattach)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_DRV_ATTACH_IN_V2_LEN);
	MCDI_DECLARE_BUF(outbuf, MC_CMD_DRV_ATTACH_EXT_OUT_LEN);
	size_t outlen = 0;
	int state, rc;

	MCDI_SET_DWORD(inbuf, DRV_ATTACH_IN_V2_NEW_STATE, new_state);
	MCDI_SET_DWORD(inbuf, DRV_ATTACH_IN_V2_UPDATE, 1);
	MCDI_SET_DWORD(inbuf, DRV_ATTACH_IN_V2_FIRMWARE_ID, fw_variant);

	strscpy(MCDI_PTR(inbuf, DRV_ATTACH_IN_V2_DRIVER_VERSION),
		EFCT_DRIVER_VERSION, MC_CMD_DRV_ATTACH_IN_V2_DRIVER_VERSION_LEN);

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_DRV_ATTACH, inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf), &outlen);

	if (!reattach && (rc || outlen < MC_CMD_DRV_ATTACH_OUT_LEN)) {
		efct_mcdi_display_error(efct, MC_CMD_DRV_ATTACH, sizeof(inbuf),
					outbuf, outlen, rc);
		if (outlen < MC_CMD_DRV_ATTACH_OUT_LEN)
			rc = -EIO;
		return rc;
	}

	state = MCDI_DWORD(outbuf, DRV_ATTACH_OUT_OLD_STATE);
	if (state != new_state)
		netif_warn(efct, probe, efct->net_dev,
			   "State set by firmware doesn't match the expected state.\n");

	if (flags && outlen >= MC_CMD_DRV_ATTACH_EXT_OUT_LEN)
		*flags = MCDI_DWORD(outbuf, DRV_ATTACH_EXT_OUT_FUNC_FLAGS);

	return rc;
}

int efct_mcdi_drv_detach(struct efct_nic *efct)
{
	return efct_mcdi_drv_attach_attempt(efct, 0, 0, NULL, false);
}

int efct_mcdi_drv_attach(struct efct_nic *efct, u32 fw_variant, u32 *out_flags,
			 bool reattach)
{
	u32 flags;
	u32 in;
	int rc;

	in = (1 << MC_CMD_DRV_ATTACH_IN_V2_ATTACH_LBN) |
	     (1 << MC_CMD_DRV_ATTACH_IN_V2_WANT_V2_LINKCHANGES_LBN);

	rc = efct_mcdi_drv_attach_attempt(efct, fw_variant, in, &flags, reattach);
	if (rc == 0) {
		pci_dbg(efct->efct_dev->pci_dev,
			"%s attached with flags %#x\n", __func__, flags);
		if (out_flags)
			*out_flags = flags;
	}

	return rc;
}

int efct_mcdi_log_ctrl(struct efct_nic *efct, bool evq, bool uart, u32 dest_evq)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_LOG_CTRL_IN_LEN);
	u32 dest = 0;
	int rc;

	if (uart)
		dest |= MC_CMD_LOG_CTRL_IN_LOG_DEST_UART;
	if (evq)
		dest |= MC_CMD_LOG_CTRL_IN_LOG_DEST_EVQ;

	MCDI_SET_DWORD(inbuf, LOG_CTRL_IN_LOG_DEST, dest);
	MCDI_SET_DWORD(inbuf, LOG_CTRL_IN_LOG_DEST_EVQ, dest_evq);

	BUILD_BUG_ON(MC_CMD_LOG_CTRL_OUT_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_LOG_CTRL, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);
	return rc;
}

/* Returns 1 if an assertion was read, 0 if no assertion had fired,
 * negative on error.
 */
static int efct_mcdi_read_assertion(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_ASSERTS_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_GET_ASSERTS_IN_LEN);
	const char *reason;
	size_t outlen = 0;
	u32 flags, index;
	int retry;
	int rc;

	/* Attempt to read any stored assertion state before we reboot
	 * the mcfw out of the assertion handler. Retry twice, once
	 * because a boot-time assertion might cause this command to fail
	 * with EINTR. And once again because GET_ASSERTS can race with
	 * MC_CMD_REBOOT running on the other port.
	 */
	retry = 2;
	do {
		MCDI_SET_DWORD(inbuf, GET_ASSERTS_IN_CLEAR, 1);
		rc = efct_mcdi_rpc_quiet(efct, MC_CMD_GET_ASSERTS,
					 inbuf, MC_CMD_GET_ASSERTS_IN_LEN,
					outbuf, sizeof(outbuf), &outlen);
		if (rc == -EPERM)
			return 0;
	} while ((rc == -EINTR || rc == -EIO) && retry-- > 0);

	if (rc) {
		efct_mcdi_display_error(efct, MC_CMD_GET_ASSERTS,
					MC_CMD_GET_ASSERTS_IN_LEN, outbuf,
				       outlen, rc);
		return rc;
	}
	if (outlen < MC_CMD_GET_ASSERTS_OUT_LEN)
		return -EIO;

	/* Print out any recorded assertion state */
	flags = MCDI_DWORD(outbuf, GET_ASSERTS_OUT_GLOBAL_FLAGS);
	if (flags == MC_CMD_GET_ASSERTS_FLAGS_NO_FAILS)
		return 0;

	reason = (flags == MC_CMD_GET_ASSERTS_FLAGS_SYS_FAIL)
		? "system-level assertion"
		: (flags == MC_CMD_GET_ASSERTS_FLAGS_THR_FAIL)
		? "thread-level assertion"
		: (flags == MC_CMD_GET_ASSERTS_FLAGS_WDOG_FIRED)
		? "watchdog reset"
		: "unknown assertion";
	netif_err(efct, hw, efct->net_dev,
		  "MCPU %s at PC = 0x%.8x in thread 0x%.8x\n", reason,
		  MCDI_DWORD(outbuf, GET_ASSERTS_OUT_SAVED_PC_OFFS),
		  MCDI_DWORD(outbuf, GET_ASSERTS_OUT_THREAD_OFFS));

	/* Print out the registers */
	for (index = 0;
	     index < MC_CMD_GET_ASSERTS_OUT_GP_REGS_OFFS_NUM;
	     index++)
		netif_err(efct, hw, efct->net_dev, "R%.2d (?): 0x%.8x\n",
			  1 + index,
			  MCDI_ARRAY_DWORD(outbuf, GET_ASSERTS_OUT_GP_REGS_OFFS,
					   index));

	return 1;
}

static int efct_mcdi_exit_assertion(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_REBOOT_IN_LEN);
	int rc;

	/* If the MC is running debug firmware, it might now be
	 * waiting for a debugger to attach, but we just want it to
	 * reboot.  We set a flag that makes the command a no-op if it
	 * has already done so.
	 * The MCDI will thus return either 0 or -EIO.
	 */
	BUILD_BUG_ON(MC_CMD_REBOOT_OUT_LEN != 0);
	MCDI_SET_DWORD(inbuf, REBOOT_IN_FLAGS,
		       MC_CMD_REBOOT_FLAGS_AFTER_ASSERTION);
	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_REBOOT, inbuf, MC_CMD_REBOOT_IN_LEN,
				 NULL, 0, NULL);
	if (rc == -EIO)
		rc = 0;
	if (rc)
		efct_mcdi_display_error(efct, MC_CMD_REBOOT, MC_CMD_REBOOT_IN_LEN,
					NULL, 0, rc);
	return rc;
}

int efct_mcdi_handle_assertion(struct efct_nic *efct)
{
	int rc;

	rc = efct_mcdi_read_assertion(efct);
	if (rc <= 0)
		return rc;

	return efct_mcdi_exit_assertion(efct);
}

static int efct_mcdi_reset_func(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_ENTITY_RESET_IN_LEN);
	int rc;

	BUILD_BUG_ON(MC_CMD_ENTITY_RESET_OUT_LEN != 0);
	MCDI_POPULATE_DWORD_1(inbuf, ENTITY_RESET_IN_FLAG,
			      ENTITY_RESET_IN_FUNCTION_RESOURCE_RESET, 1);
	rc = efct_mcdi_rpc(efct, MC_CMD_ENTITY_RESET, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);
	return rc;
}

static int efct_mcdi_reset_mc(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_REBOOT_IN_LEN);
	int rc;

	BUILD_BUG_ON(MC_CMD_REBOOT_OUT_LEN != 0);
	MCDI_SET_DWORD(inbuf, REBOOT_IN_FLAGS, 0);
	rc = efct_mcdi_rpc(efct, MC_CMD_REBOOT, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);
	/* White is black, and up is down */
	if (rc == -EIO)
		return 0;
	if (rc == 0)
		rc = -EIO;
	return rc;
}

int efct_flr(struct efct_nic *efct)
{
	struct efct_mcdi_iface *mcdi = efct_mcdi(efct);
	LIST_HEAD(cleanup_list);
	u16 seq;
	int rc;

	netif_dbg(efct, drv, efct->net_dev, "Beginning FLR\n");

	rc = pci_reset_function(efct->efct_dev->pci_dev);
	if (rc)
		return rc;

	if (!mcdi)
		return 0;

	spin_lock_bh(&mcdi->iface_lock);
	while (!list_empty(&mcdi->cmd_list)) {
		struct efct_mcdi_cmd *cmd =
			list_first_entry(&mcdi->cmd_list,
					 struct efct_mcdi_cmd, list);

		netif_dbg(efct, drv, efct->net_dev,
			  "aborting command %#x inlen %zu due to FLR\n",
			  cmd->cmd, cmd->inlen);

		kref_get(&cmd->ref);

		cmd->rc = -EIO;

		if (efct_cmd_running(cmd))
			efct->type->mcdi_put_buf(efct, cmd->bufid);

		efct_mcdi_remove_cmd(mcdi, cmd, &cleanup_list);

		if (cancel_delayed_work(&cmd->work))
			kref_put(&cmd->ref, efct_mcdi_cmd_release);

		kref_put(&cmd->ref, efct_mcdi_cmd_release);
	}

	mcdi->db_held_by = NULL;
	for (seq = 0; seq < ARRAY_SIZE(mcdi->seq_held_by); ++seq)
		mcdi->seq_held_by[seq] = NULL;
	mcdi->mode = MCDI_MODE_POLL;

	spin_unlock_bh(&mcdi->iface_lock);

	netif_dbg(efct, drv, efct->net_dev, "Cleaning up for FLR\n");

	efct_mcdi_process_cleanup_list(efct, &cleanup_list);

	netif_dbg(efct, drv, efct->net_dev, "FLR complete\n");

	return 0;
}

int efct_mcdi_reset(struct efct_nic *efct, enum reset_type method)
{
	int rc;

	/* Recover from a failed assertion pre-reset */
	rc = efct_mcdi_handle_assertion(efct);
	if (rc) {
		netif_err(efct, probe, efct->net_dev,
			  "Unable to handle assertion\n");
		return rc;
	}

	if (method == RESET_TYPE_DATAPATH || method == RESET_TYPE_MC_BIST)
		rc = 0;
	else if (method == RESET_TYPE_WORLD)
		rc = efct_mcdi_reset_mc(efct);
	else
		rc = efct_mcdi_reset_func(efct);

	return rc;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_NEED_ETHTOOL_EROM_VERSION)
void efct_mcdi_erom_ver(struct efct_nic *efct,
			char *buf,
			size_t len)
{
	u16 version[4];
	int rc;

	rc = efct_mcdi_nvram_metadata(efct, NVRAM_PARTITION_TYPE_EXPANSION_ROM,
				      NULL, version, NULL, 0);
	if (rc)
		return;
	len = min_t(size_t, EFCT_MAX_VERSION_INFO_LEN, len);
	snprintf(buf, len, "%u.%u.%u.%u", version[0],
		 version[1], version[2], version[3]);
}
#endif

void efct_mcdi_print_fwver(struct efct_nic *efct, char *buf, size_t len)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_GET_VERSION_OUT_LEN);
	const __le16 *ver_words;
	size_t outlength;
	int rc;

	BUILD_BUG_ON(MC_CMD_GET_VERSION_IN_LEN != 0);
	rc = efct_mcdi_rpc(efct, MC_CMD_GET_VERSION, NULL, 0,
			   outbuf, sizeof(outbuf), &outlength);
	if (rc)
		goto fail;
	if (outlength < MC_CMD_GET_VERSION_OUT_LEN)
		goto fail;

	ver_words = (__le16 *)MCDI_PTR(outbuf, GET_VERSION_OUT_VERSION);
	snprintf(buf, len, "%u.%u.%u.%u",
		 le16_to_cpu(ver_words[0]), le16_to_cpu(ver_words[1]),
		 le16_to_cpu(ver_words[2]), le16_to_cpu(ver_words[3]));

	return;

fail:
	buf[0] = 0;
}

#define EFCT_MCDI_NVRAM_LEN_MAX 128

int efct_mcdi_nvram_update_start(struct efct_nic *efct, u32 type)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_UPDATE_START_V2_IN_LEN);
	int rc;

	MCDI_SET_DWORD(inbuf, NVRAM_UPDATE_START_IN_TYPE, type);
	MCDI_POPULATE_DWORD_1(inbuf, NVRAM_UPDATE_START_V2_IN_FLAGS,
			      NVRAM_UPDATE_START_V2_IN_FLAG_REPORT_VERIFY_RESULT, 1);

	BUILD_BUG_ON(MC_CMD_NVRAM_UPDATE_START_OUT_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_UPDATE_START, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);

	return rc;
}

int efct_mcdi_nvram_write(struct efct_nic *efct, u32 type,
			  loff_t offset, const u8 *buffer, size_t length)
{
	efct_dword_t *inbuf;
	size_t inlen;
	int rc;

	inlen = ALIGN(MC_CMD_NVRAM_WRITE_IN_LEN(length), 4);
	inbuf = kzalloc(inlen, GFP_KERNEL);
	if (!inbuf)
		return -ENOMEM;

	MCDI_SET_DWORD(inbuf, NVRAM_WRITE_IN_TYPE, type);
	MCDI_SET_DWORD(inbuf, NVRAM_WRITE_IN_OFFSET, offset);
	MCDI_SET_DWORD(inbuf, NVRAM_WRITE_IN_LENGTH, length);
	memcpy(MCDI_PTR(inbuf, NVRAM_WRITE_IN_WRITE_BUFFER), buffer, length);

	BUILD_BUG_ON(MC_CMD_NVRAM_WRITE_OUT_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_WRITE, inbuf, inlen, NULL, 0, NULL);
	kfree(inbuf);

	return rc;
}

int efct_mcdi_nvram_erase(struct efct_nic *efct, u32 type,
			  loff_t offset, size_t length)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_ERASE_IN_LEN);
	int rc;

	MCDI_SET_DWORD(inbuf, NVRAM_ERASE_IN_TYPE, type);
	MCDI_SET_DWORD(inbuf, NVRAM_ERASE_IN_OFFSET, offset);
	MCDI_SET_DWORD(inbuf, NVRAM_ERASE_IN_LENGTH, length);

	BUILD_BUG_ON(MC_CMD_NVRAM_ERASE_OUT_LEN != 0);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_ERASE, inbuf, sizeof(inbuf),
			   NULL, 0, NULL);
	return rc;
}

int efct_mcdi_nvram_update_finish(struct efct_nic *efct, u32 type,
				  enum efct_update_finish_mode mode)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_UPDATE_FINISH_V2_IN_LEN);
	size_t outlen;
	int rc, rc2;
	u32 reboot;

	/* Reboot PHY's into the new firmware. mcfw reboot is handled
	 * explicity via ethtool.
	 */
	reboot = (type == MC_CMD_NVRAM_TYPE_PHY_PORT0 ||
		  type == MC_CMD_NVRAM_TYPE_PHY_PORT1 ||
		  type == MC_CMD_NVRAM_TYPE_DISABLED_CALLISTO);
	MCDI_SET_DWORD(inbuf, NVRAM_UPDATE_FINISH_IN_TYPE, type);
	MCDI_SET_DWORD(inbuf, NVRAM_UPDATE_FINISH_IN_REBOOT, reboot);

	/* Old firmware doesn't support background update finish and abort
	 * operations. Fallback to waiting if the requested mode is not
	 * supported.
	 */
	if (!efct_has_cap(efct, NVRAM_UPDATE_POLL_VERIFY_RESULT) ||
	    (!efct_has_cap(efct, NVRAM_UPDATE_ABORT_SUPPORTED) &&
	     mode == EFCT_UPDATE_FINISH_ABORT))
		mode = EFCT_UPDATE_FINISH_WAIT;

	MCDI_POPULATE_DWORD_4(inbuf, NVRAM_UPDATE_FINISH_V2_IN_FLAGS,
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_REPORT_VERIFY_RESULT,
			      (mode != EFCT_UPDATE_FINISH_ABORT),
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_RUN_IN_BACKGROUND,
			      (mode == EFCT_UPDATE_FINISH_BACKGROUND),
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_POLL_VERIFY_RESULT,
			      (mode == EFCT_UPDATE_FINISH_POLL),
			      NVRAM_UPDATE_FINISH_V2_IN_FLAG_ABORT,
			      (mode == EFCT_UPDATE_FINISH_ABORT));

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_UPDATE_FINISH, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (!rc && outlen >= MC_CMD_NVRAM_UPDATE_FINISH_V2_OUT_LEN) {
		rc2 = MCDI_DWORD(outbuf, NVRAM_UPDATE_FINISH_V2_OUT_RESULT_CODE);
		if (rc2 != MC_CMD_NVRAM_VERIFY_RC_SUCCESS &&
		    rc2 != MC_CMD_NVRAM_VERIFY_RC_PENDING)
			netif_err(efct, drv, efct->net_dev,
				  "NVRAM update failed verification with code 0x%x\n",
				  rc2);
		switch (rc2) {
		case MC_CMD_NVRAM_VERIFY_RC_SUCCESS:
			break;
		case MC_CMD_NVRAM_VERIFY_RC_PENDING:
			rc = -EAGAIN;
			break;
		case MC_CMD_NVRAM_VERIFY_RC_CMS_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_MESSAGE_DIGEST_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_TRUSTED_APPROVERS_CHECK_FAILED:
		case MC_CMD_NVRAM_VERIFY_RC_SIGNATURE_CHAIN_CHECK_FAILED:
			rc = -EIO;
			break;
		case MC_CMD_NVRAM_VERIFY_RC_INVALID_CMS_FORMAT:
		case MC_CMD_NVRAM_VERIFY_RC_BAD_MESSAGE_DIGEST:
			rc = -EINVAL;
			break;
		case MC_CMD_NVRAM_VERIFY_RC_NO_VALID_SIGNATURES:
		case MC_CMD_NVRAM_VERIFY_RC_NO_TRUSTED_APPROVERS:
		case MC_CMD_NVRAM_VERIFY_RC_NO_SIGNATURE_MATCH:
		case MC_CMD_NVRAM_VERIFY_RC_REJECT_TEST_SIGNED:
		case MC_CMD_NVRAM_VERIFY_RC_SECURITY_LEVEL_DOWNGRADE:
			rc = -EPERM;
			break;
		default:
			netif_err(efct, drv, efct->net_dev,
				  "Unknown response to NVRAM_UPDATE_FINISH\n");
			rc = -EIO;
		}
	}
	return rc;
}

#define	EFCT_MCDI_NVRAM_UPDATE_FINISH_INITIAL_POLL_DELAY_MS 5
#define	EFCT_MCDI_NVRAM_UPDATE_FINISH_MAX_POLL_DELAY_MS 5000
#define	EFCT_MCDI_NVRAM_UPDATE_FINISH_RETRIES 185

int efct_mcdi_nvram_update_finish_polled(struct efct_nic *efct, u32 type)
{
	u32 delay = EFCT_MCDI_NVRAM_UPDATE_FINISH_INITIAL_POLL_DELAY_MS;
	u32 retry = 0;
	int rc;

	/* NVRAM updates can take a long time (e.g. up to 1 minute for bundle
	 * images). Polling for NVRAM update completion ensures that other MCDI
	 * commands can be issued before the background NVRAM update completes.
	 *
	 * The initial call either completes the update synchronously, or
	 * returns -EAGAIN to indicate processing is continuing. In the latter
	 * case, we poll for at least 900 seconds, at increasing intervals
	 * (5ms, 50ms, 500ms, 5s).
	 */
	rc = efct_mcdi_nvram_update_finish(efct, type, EFCT_UPDATE_FINISH_BACKGROUND);
	while (rc == -EAGAIN) {
		if (retry > EFCT_MCDI_NVRAM_UPDATE_FINISH_RETRIES)
			return -ETIMEDOUT;
		retry++;

		msleep(delay);
		if (delay < EFCT_MCDI_NVRAM_UPDATE_FINISH_MAX_POLL_DELAY_MS)
			delay *= 10;

		rc = efct_mcdi_nvram_update_finish(efct, type, EFCT_UPDATE_FINISH_POLL);
	}
	return rc;
}

int efct_mcdi_nvram_metadata(struct efct_nic *efct, u32 type,
			     u32 *subtype, u16 version[4], char *desc,
			    size_t descsize)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_METADATA_IN_LEN);
	efct_dword_t *outbuf;
	size_t outlen;
	u32 flags;
	int rc;

	outbuf = kzalloc(MC_CMD_NVRAM_METADATA_OUT_LENMAX_MCDI2, GFP_KERNEL);
	if (!outbuf)
		return -ENOMEM;

	MCDI_SET_DWORD(inbuf, NVRAM_METADATA_IN_TYPE, type);

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_NVRAM_METADATA, inbuf,
				 sizeof(inbuf), outbuf,
				MC_CMD_NVRAM_METADATA_OUT_LENMAX_MCDI2,
				&outlen);
	if (rc)
		goto out_free;
	if (outlen < MC_CMD_NVRAM_METADATA_OUT_LENMIN) {
		rc = -EIO;
		goto out_free;
	}

	flags = MCDI_DWORD(outbuf, NVRAM_METADATA_OUT_FLAGS);

	if (desc && descsize > 0) {
		if (flags & BIT(MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_VALID_LBN)) {
			if (descsize <=
			    MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_NUM(outlen)) {
				rc = -E2BIG;
				goto out_free;
			}

			strncpy(desc,
				MCDI_PTR(outbuf, NVRAM_METADATA_OUT_DESCRIPTION),
				MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_NUM(outlen));
			desc[MC_CMD_NVRAM_METADATA_OUT_DESCRIPTION_NUM(outlen)] = '\0';
		} else {
			desc[0] = '\0';
		}
	}

	if (subtype) {
		if (flags & BIT(MC_CMD_NVRAM_METADATA_OUT_SUBTYPE_VALID_LBN))
			*subtype = MCDI_DWORD(outbuf, NVRAM_METADATA_OUT_SUBTYPE);
		else
			*subtype = 0;
	}

	if (version) {
		if (flags & BIT(MC_CMD_NVRAM_METADATA_OUT_VERSION_VALID_LBN)) {
			version[0] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_W);
			version[1] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_X);
			version[2] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_Y);
			version[3] = MCDI_WORD(outbuf, NVRAM_METADATA_OUT_VERSION_Z);
		} else {
			version[0] = 0;
			version[1] = 0;
			version[2] = 0;
			version[3] = 0;
		}
	}

out_free:
	kfree(outbuf);
	return rc;
}

#define EFCT_MCDI_NVRAM_DEFAULT_WRITE_LEN 128

int efct_mcdi_nvram_info(struct efct_nic *efct, u32 type,
			 size_t *size_out, size_t *erase_size_out,
			size_t *write_size_out, bool *protected_out)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_NVRAM_INFO_V2_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_NVRAM_INFO_IN_LEN);
	size_t write_size = 0;
	size_t outlen;
	int rc;

	MCDI_SET_DWORD(inbuf, NVRAM_INFO_IN_TYPE, type);

	rc = efct_mcdi_rpc(efct, MC_CMD_NVRAM_INFO, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc)
		goto fail;
	if (outlen < MC_CMD_NVRAM_INFO_OUT_LEN) {
		rc = -EIO;
		goto fail;
	}

	if (outlen >= MC_CMD_NVRAM_INFO_V2_OUT_LEN)
		write_size = MCDI_DWORD(outbuf, NVRAM_INFO_V2_OUT_WRITESIZE);
	else
		write_size = EFCT_MCDI_NVRAM_DEFAULT_WRITE_LEN;

	*write_size_out = write_size;
	*size_out = MCDI_DWORD(outbuf, NVRAM_INFO_OUT_SIZE);
	*erase_size_out = MCDI_DWORD(outbuf, NVRAM_INFO_OUT_ERASESIZE);
	*protected_out = !!(MCDI_DWORD(outbuf, NVRAM_INFO_OUT_FLAGS) &
				(1 << MC_CMD_NVRAM_INFO_OUT_PROTECTED_LBN));
	return 0;

fail:
	netif_err(efct, hw, efct->net_dev, "%s: failed rc=%d\n", __func__, rc);
	return rc;
}
