// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include "net_driver.h"
#include "nic.h"
#include "mcdi_functions.h"
#include "mcdi.h"
#include "efct_nic.h"

int efct_mcdi_ev_init(struct efct_ev_queue *eventq)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_INIT_EVQ_V3_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_INIT_EVQ_V3_IN_LEN);
	struct efct_nic *efct = eventq->efct;
	dma_addr_t dma_addr;
	size_t outlen;
	int rc;

	/* Fill event queue with all ones (i.e. empty events) */
	memset(eventq->buf.addr, 0xff, eventq->buf.len);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_SIZE, eventq->entries);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_INSTANCE, eventq->index);
	/* INIT_EVQ expects index in vector table, not absolute */
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_IRQ_NUM, eventq->index);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TMR_MODE, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TMR_LOAD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TMR_RELOAD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_COUNT_MODE, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_COUNT_THRSHLD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_FLAG_INT_ARMD, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_FLAG_RPTR_DOS, 0);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_RX_MERGE_TIMEOUT_NS, eventq->rx_merge_timeout_ns);
	MCDI_SET_DWORD(inbuf, INIT_EVQ_V3_IN_TX_MERGE_TIMEOUT_NS, eventq->tx_merge_timeout_ns);

	MCDI_POPULATE_DWORD_6(inbuf, INIT_EVQ_V3_IN_FLAGS,
			      INIT_EVQ_V3_IN_FLAG_INTERRUPTING, 1,
			      INIT_EVQ_V3_IN_FLAG_RX_MERGE, 1,
			      INIT_EVQ_V3_IN_FLAG_TX_MERGE, 1,
			      INIT_EVQ_V3_IN_FLAG_TYPE, 0,
			      INIT_EVQ_V3_IN_FLAG_USE_TIMER, 1,
			      INIT_EVQ_V3_IN_FLAG_CUT_THRU, 0);

	dma_addr = eventq->buf.dma_addr;
	MCDI_SET_QWORD(inbuf, INIT_EVQ_V3_IN_DMA_ADDR, dma_addr);
	rc = efct_mcdi_rpc(efct, MC_CMD_INIT_EVQ, inbuf, MC_CMD_INIT_EVQ_V3_IN_LEN,
			   outbuf, sizeof(outbuf), &outlen);

	if (outlen >= MC_CMD_INIT_EVQ_V3_OUT_LEN)
		netif_dbg(efct, drv, efct->net_dev,
			  "Index %d using event queue flags %08x\n",
			  eventq->index,
			  MCDI_DWORD(outbuf, INIT_EVQ_V3_OUT_FLAGS));

	return rc;
}

int efct_mcdi_ev_set_timer(struct efct_ev_queue *eventq, u32 ns, u32 mode, bool async)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_SET_EVQ_TMR_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_SET_EVQ_TMR_IN_LEN);
	struct efct_nic *efct = eventq->efct;
	size_t outlen;
	int rc;

	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_INSTANCE, eventq->index);
	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_TMR_LOAD_REQ_NS, ns);
	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_TMR_RELOAD_REQ_NS, ns);
	MCDI_SET_DWORD(inbuf, SET_EVQ_TMR_IN_TMR_MODE, mode);

	if (async) {
		rc = efct_mcdi_rpc_async(efct, MC_CMD_SET_EVQ_TMR, inbuf, sizeof(inbuf), NULL, 0);
	} else {
		rc = efct_mcdi_rpc(efct, MC_CMD_SET_EVQ_TMR, inbuf, sizeof(inbuf),
				   outbuf, sizeof(outbuf), &outlen);
		if (rc)
			efct_mcdi_display_error(efct, MC_CMD_SET_EVQ_TMR,
						MC_CMD_SET_EVQ_TMR_IN_LEN, NULL, 0, rc);
		else
			/* Saving the actual value set */
			eventq->irq_moderation_ns = MCDI_DWORD(outbuf,
							       SET_EVQ_TMR_OUT_TMR_RELOAD_ACT_NS);
	}

	return rc;
}

int efct_mcdi_ev_fini(struct efct_ev_queue *eventq)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FINI_EVQ_IN_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	size_t outlen;
	struct efct_nic *efct;
	int rc;

	if (!eventq || !eventq->efct)
		return -EINVAL;

	efct = eventq->efct;
	MCDI_SET_DWORD(inbuf, FINI_EVQ_IN_INSTANCE, eventq->index);

	rc = efct_mcdi_rpc(efct, MC_CMD_FINI_EVQ, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc && rc != -EALREADY)
		efct_mcdi_display_error(efct, MC_CMD_FINI_EVQ, MC_CMD_FINI_EVQ_IN_LEN, NULL, 0, rc);

	return rc;
}

int efct_mcdi_rx_init(struct efct_rx_queue *rx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_INIT_RXQ_V5_IN_LEN);
	struct efct_nic *efct = rx_queue->efct;
	int rc;

	BUILD_BUG_ON(MC_CMD_INIT_RXQ_V5_OUT_LEN != 0);
	/*set the inbuf memory to zero*/
	memset(inbuf, 0, MC_CMD_INIT_RXQ_V5_IN_LEN);

	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_SIZE, rx_queue->num_entries);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_TARGET_EVQ, rx_queue->evq_index);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_LABEL, rx_queue->label);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_INSTANCE, rx_queue->index);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_PORT_ID, EVB_PORT_ID_ASSIGNED);
	MCDI_POPULATE_DWORD_4(inbuf, INIT_RXQ_V5_IN_FLAGS,
			      INIT_RXQ_V5_IN_DMA_MODE,
			      MC_CMD_INIT_RXQ_V5_IN_EQUAL_STRIDE_SUPER_BUFFER,
			      INIT_RXQ_V5_IN_FLAG_TIMESTAMP, 1,
			      INIT_RXQ_V5_IN_FLAG_PREFIX, 1,
			      INIT_RXQ_V5_IN_FLAG_DISABLE_SCATTER, 1);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_ES_PACKET_STRIDE, roundup_pow_of_two(efct->mtu));
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_ES_MAX_DMA_LEN, efct->mtu);
	MCDI_SET_DWORD(inbuf, INIT_RXQ_V5_IN_ES_PACKET_BUFFERS_PER_BUCKET,
		       DIV_ROUND_UP(rx_queue->buffer_size, rx_queue->pkt_stride));

	rc = efct_mcdi_rpc(efct, MC_CMD_INIT_RXQ, inbuf,
			   MC_CMD_INIT_RXQ_V5_IN_LEN, NULL, 0, NULL);
	if (rc && rc != -ENETDOWN && rc != -EAGAIN)
		netif_err(efct, ifup, efct->net_dev,
			  "failed to initialise RXQ %d, error %d\n", rx_queue->index, rc);

	return rc;
}

int efct_mcdi_rx_fini(struct efct_rx_queue *rx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FINI_RXQ_IN_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	struct efct_nic *efct = rx_queue->efct;
	size_t outlen;
	int rc;

	MCDI_SET_DWORD(inbuf, FINI_RXQ_IN_INSTANCE, rx_queue->index);

	rc = efct_mcdi_rpc(efct, MC_CMD_FINI_RXQ, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc && rc != -EALREADY)
		efct_mcdi_display_error(efct, MC_CMD_FINI_RXQ, MC_CMD_FINI_RXQ_IN_LEN,
					outbuf, outlen, rc);

	return rc;
}

int efct_mcdi_tx_init(struct efct_tx_queue *tx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_INIT_TXQ_EXT_IN_LEN);
	struct efct_nic *efct = tx_queue->efct;
	int rc;

	BUILD_BUG_ON(MC_CMD_INIT_TXQ_OUT_LEN != 0);
	/*set the inbuf memory to zero*/
	memset(inbuf, 0, MC_CMD_INIT_TXQ_EXT_IN_LEN);

	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_TARGET_EVQ, tx_queue->evq_index);
	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_LABEL, tx_queue->label);
	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_INSTANCE, tx_queue->txq_index);
	MCDI_SET_DWORD(inbuf, INIT_TXQ_EXT_IN_PORT_ID, EVB_PORT_ID_ASSIGNED);
	//TBD crc mode
	MCDI_POPULATE_DWORD_4(inbuf, INIT_TXQ_EXT_IN_FLAGS,
			      INIT_TXQ_EXT_IN_FLAG_IP_CSUM_DIS, 1,
			      INIT_TXQ_EXT_IN_FLAG_TCP_CSUM_DIS, 1,
			      INIT_TXQ_EXT_IN_FLAG_CTPIO, 1,
			      INIT_TXQ_EXT_IN_FLAG_CTPIO_UTHRESH, 1);

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_INIT_TXQ,
				 inbuf, sizeof(inbuf),
				NULL, 0, NULL);
	if (rc) {
		efct_mcdi_display_error(efct, MC_CMD_INIT_TXQ,
					MC_CMD_INIT_TXQ_EXT_IN_LEN,
				       NULL, 0, rc);
		return rc;
	}

	return 0;
}

int efct_mcdi_tx_fini(struct efct_tx_queue *tx_queue)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FINI_TXQ_IN_LEN);
	MCDI_DECLARE_BUF_ERR(outbuf);
	struct efct_nic *efct;
	size_t outlen;
	int rc;

	efct = tx_queue->efct;
	MCDI_SET_DWORD(inbuf, FINI_TXQ_IN_INSTANCE, tx_queue->txq_index);

	rc = efct_mcdi_rpc(efct, MC_CMD_FINI_TXQ, inbuf, sizeof(inbuf),
			   outbuf, sizeof(outbuf), &outlen);
	if (rc && rc != -EALREADY)
		efct_mcdi_display_error(efct, MC_CMD_FINI_TXQ, MC_CMD_FINI_TXQ_IN_LEN,
					outbuf, outlen, rc);

	return rc;
}

int efct_mcdi_filter_insert(struct efct_nic *efct, struct efct_filter_spec *rule, u64 *handle)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_FILTER_OP_EXT_OUT_LEN);
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FILTER_OP_V3_IN_LEN);
	size_t outlen;
	int rc;

	memset(inbuf, 0, MC_CMD_FILTER_OP_V3_IN_LEN);
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_OP, MC_CMD_FILTER_OP_IN_OP_INSERT);

	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_RX_MODE, MC_CMD_FILTER_OP_IN_RX_MODE_SIMPLE);
	MCDI_SET_DWORD(inbuf, FILTER_OP_IN_RX_DEST,
		       (rule->queue_id == RX_CLS_FLOW_DISC ?
		       MC_CMD_FILTER_OP_IN_RX_DEST_DROP :
		       MC_CMD_FILTER_OP_IN_RX_DEST_HOST));
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_TX_DEST, MC_CMD_FILTER_OP_IN_TX_DEST_DEFAULT);
	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_PORT_ID, EVB_PORT_ID_ASSIGNED);

	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_MATCH_FIELDS, rule->match_fields);
	MCDI_SET_DWORD(inbuf, FILTER_OP_IN_RX_QUEUE,
		       (rule->queue_id == RX_CLS_FLOW_DISC ?
		       0 : rule->queue_id));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_DST_IP), &rule->dst_ip, sizeof(rule->dst_ip));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_DST_PORT), &rule->dst_port, sizeof(rule->dst_port));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_IP_PROTO), &rule->ip_proto, sizeof(rule->ip_proto));
	memcpy(MCDI_PTR(inbuf, FILTER_OP_V3_IN_ETHER_TYPE), &rule->ether_type,
	       sizeof(rule->ether_type));

	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_FILTER_OP, inbuf, sizeof(inbuf),
				 outbuf, sizeof(outbuf), &outlen);
	if (rc == 0)
		*handle = MCDI_QWORD(outbuf, FILTER_OP_OUT_HANDLE);

	return rc;
}

int efct_mcdi_filter_remove(struct efct_nic *efct, u64 handle)
{
	MCDI_DECLARE_BUF(inbuf, MC_CMD_FILTER_OP_V3_IN_LEN);
	int rc;

	MCDI_SET_DWORD(inbuf, FILTER_OP_V3_IN_OP, MC_CMD_FILTER_OP_IN_OP_REMOVE);
	MCDI_SET_QWORD(inbuf, FILTER_OP_V3_IN_HANDLE, handle);
	rc = efct_mcdi_rpc_quiet(efct, MC_CMD_FILTER_OP, inbuf,
				 sizeof(inbuf), NULL, 0, NULL);
	return rc;
}

int efct_mcdi_filter_table_probe(struct efct_nic *efct)
{
	struct efct_mcdi_filter_table *table;
	int rc = 0, i;

	if (efct->filter_table) /* already probed */
		return rc;

	table = kzalloc(sizeof(*table), GFP_KERNEL);
	if (!table)
		return -ENOMEM;

	efct->filter_table = table;

	init_rwsem(&table->lock);
	table->entry = vzalloc(EFCT_MCDI_FILTER_TBL_ROWS *
			       sizeof(*table->entry));
	if (!table->entry) {
		rc = -ENOMEM;
		return rc;
	}

	for (i = 0; i < EFCT_MCDI_FILTER_TBL_ROWS; i++) {
		table->entry[i].handle = EFCT_HANDLE_INVALID;
		table->entry[i].ref_cnt = 0;
	}

	return rc;
}

void efct_mcdi_filter_table_remove(struct efct_nic *efct)
{
	struct efct_mcdi_filter_table *table = efct->filter_table;
	int i;

	if (!table)
		return;
	for (i = 0; i < EFCT_MCDI_FILTER_TBL_ROWS; i++) {
		if (table->entry[i].spec)
			kfree((struct efct_filter_spec *)table->entry[i].spec);
	}
	vfree(table->entry);
	table->entry = NULL;
	efct->filter_table = NULL;
	kfree(table);
}
