// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/bitops.h>
#include <linux/slab.h>
#include <linux/hwmon.h>
#include <linux/stat.h>

#include "net_driver.h"
#include "mcdi_pcol.h"
#include "nic.h"
#include "mcdi.h"

#define EFCT_HWMON_TYPES_COUNT	hwmon_pwm

#define EFCT_DYNAMIC_SENSOR_READING_UPDATE_MAX_HANDLES 32
#define EFCT_DYNAMIC_SENSOR_INFO_READ_MAX_HANDLES 4

#define EFCT_HWMON_ATTRIBUTE_COUNT	(EFCT_HWMON_NAME + 1)

#define       MC_CMD_DYN_SENSOR_LIMIT_LO_WARNING_OFST \
		(MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_OFST + \
				MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_WARNING_OFST)
#define       MC_CMD_DYN_SENSOR_LIMIT_LO_CRITICAL_OFST \
		(MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_OFST + \
				MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_CRITICAL_OFST)
#define       MC_CMD_DYN_SENSOR_LIMIT_LO_FATAL_OFST \
		(MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_OFST + \
				MC_CMD_DYNAMIC_SENSORS_LIMITS_LO_FATAL_OFST)
#define       MC_CMD_DYN_SENSOR_LIMIT_HI_WARNING_OFST \
		(MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_OFST + \
				MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_WARNING_OFST)
#define       MC_CMD_DYN_SENSOR_LIMIT_HI_CRITICAL_OFST \
		(MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_OFST + \
				MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_CRITICAL_OFST)
#define       MC_CMD_DYN_SENSOR_LIMIT_HI_FATAL_OFST \
		(MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_LIMITS_OFST + \
				MC_CMD_DYNAMIC_SENSORS_LIMITS_HI_FATAL_OFST)

struct efct_mcdi_hwmon_info {
	const char *label;
	enum hwmon_sensor_types hwmon_type;
	int port;
};

enum efct_hwmon_attribute {
	EFCT_HWMON_INPUT,
	EFCT_HWMON_WARNING_LOW,
	EFCT_HWMON_WARNING_HIGH,
	EFCT_HWMON_CRIT_LOW,
	EFCT_HWMON_CRIT_HIGH,
	EFCT_HWMON_ALARM,
	EFCT_HWMON_LABEL,
	EFCT_HWMON_NAME
};

static const char *const sensor_status_names[] = {
	[MC_CMD_DYNAMIC_SENSORS_READING_OK] = "OK",
	[MC_CMD_DYNAMIC_SENSORS_READING_WARNING] = "Warning",
	[MC_CMD_DYNAMIC_SENSORS_READING_CRITICAL] = "Critical",
	[MC_CMD_DYNAMIC_SENSORS_READING_FATAL] = "Fatal",
	[MC_CMD_DYNAMIC_SENSORS_READING_BROKEN] = "Device Failure",
	[MC_CMD_DYNAMIC_SENSORS_READING_NO_READING] = "No reading",
	[MC_CMD_DYNAMIC_SENSORS_READING_INIT_FAILED] = "Initialization Failed",
};

struct efct_mcdi_mon_attribute {
	struct attribute dev_attr;
	struct sysfs_ops sensorops;
	u32 index;
	u32 type;
	enum hwmon_sensor_types hwmon_type;
	u32 limit_value;
	enum efct_hwmon_attribute hwmon_attribute;
	u8 file_index;
#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_HWMON_DEVICE_REGISTER_WITH_INFO)
	char name[16];
#endif
};

#ifdef CONFIG_XILINX_MCDI_MON

enum efct_sensor_limits {
	EFCT_SENSOR_LIMIT_WARNING_LO,
	EFCT_SENSOR_LIMIT_CRITICAL_LO,
	EFCT_SENSOR_LIMIT_FATAL_LO,
	EFCT_SENSOR_LIMIT_WARNING_HI,
	EFCT_SENSOR_LIMIT_CRITICAL_HI,
	EFCT_SENSOR_LIMIT_FATAL_HI,
	EFCT_SENSOR_LIMITS
};

/* struct efct_dynamic_sensor_description - dynamic sensor description
 * @name: name of the sensor
 * @idx: sensor index in the host driver maintained list
 * @handle: handle to the sensor
 * @type: type of sensor
 * @limits: limits for the sensor reading
 * @gen_count: generation count corresponding to the description
 * @entry: an entry in rhashtable of dynamic sensors
 */
struct efct_dynamic_sensor_description {
	char name[MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_NAME_LEN];
	u32 idx;
	u32 handle;
	u32 type;
	int limits[EFCT_SENSOR_LIMITS];
	u32 gen_count;
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_RHASHTABLE_LOOKUP_FAST)
	struct rhash_head entry;
#endif
};

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_RHASHTABLE_LOOKUP_FAST)
static const struct rhashtable_params sensor_entry_params = {
	.key_len     = sizeof(u32),
	.key_offset  = offsetof(struct efct_dynamic_sensor_description, handle),
	.head_offset = offsetof(struct efct_dynamic_sensor_description, entry),
};
#endif

static struct efct_dynamic_sensor_description *
efct_mcdi_get_dynamic_sensor(struct efct_nic *efct, u32 handle)
{
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_RHASHTABLE_LOOKUP_FAST)
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);

	return rhashtable_lookup_fast(&hwmon->sensor_table, &handle,
				      sensor_entry_params);
#else
	return NULL;
#endif
}

static void
efct_mcdi_mon_add_attr(struct efct_nic *efct,
		       u32 index, u32 type,
		      u32 limit_value, u8 file_index,
		      enum efct_hwmon_attribute attribute)
{
	struct efct_mcdi_mon_attribute *attr;
	struct efct_mcdi_mon *hwmon;

	hwmon = efct_mcdi_mon(efct);
	attr = &hwmon->attrs[hwmon->n_attrs];
	attr->index = index;
	attr->type = type;

	/* Conversion between FW types and kernel types */
	if (type == 0)
		attr->hwmon_type = hwmon_in;
	if (type == 1)
		attr->hwmon_type = hwmon_curr;
	if (type == 2)
		attr->hwmon_type = hwmon_power;
	if (type == 3)
		attr->hwmon_type = hwmon_temp;
	if (type == 4)
		attr->hwmon_type = hwmon_fan;

	attr->limit_value = limit_value;
	attr->file_index = file_index;
	attr->hwmon_attribute = attribute;
	++hwmon->n_attrs;
}

#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_RHASHTABLE_LOOKUP_FAST)
static void efct_mcdi_add_dynamic_sensor(struct efct_nic *efct,
					 u32 handle,
					u32 idx)
{
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	struct efct_dynamic_sensor_description *sensor;

	sensor = (struct efct_dynamic_sensor_description *)hwmon->sensor_list;
	sensor += idx;
	/* sensor->idx need to stored as this is needed to access sensor
	 * reading data saved
	 */
	sensor->idx = idx;
	sensor->handle = handle;
	rhashtable_lookup_insert_fast(&hwmon->sensor_table,
				      &sensor->entry,
				      sensor_entry_params);
}

static void efct_mcdi_remove_dynamic_sensors(struct efct_nic *efct)
{
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	struct efct_dynamic_sensor_description *sensor;
	int i = 0;

	sensor = (struct efct_dynamic_sensor_description *)hwmon->sensor_list;
	for (i = 0; i < hwmon->n_dynamic_sensors; i++) {
		rhashtable_remove_fast(&hwmon->sensor_table,
				       &sensor[i].entry,
				       sensor_entry_params);
	}
}

/* This function is assumed to be called only after @has_dynamic_sensors
 * has returned true or DYNAMIC_SENSOR_LIST event received
 */
static int efct_mcdi_read_dynamic_sensor_list(struct efct_nic *efct)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_DYNAMIC_SENSORS_LIST_OUT_LENMAX);
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	u32 n_sensors;
	u32 gen_count;
	size_t outlen;
	int rc, i;

	rc = efct_mcdi_rpc(efct, MC_CMD_DYNAMIC_SENSORS_LIST, NULL, 0, outbuf,
			   sizeof(outbuf), &outlen);
	if (rc)
		return rc;

	gen_count = MCDI_DWORD(outbuf, DYNAMIC_SENSORS_LIST_OUT_GENERATION);
	/* check if generation count changed */
	if (gen_count == hwmon->generation_count)
		return 0;
	hwmon->generation_count = gen_count;

	n_sensors = MCDI_DWORD(outbuf, DYNAMIC_SENSORS_LIST_OUT_COUNT);
	if (outlen < MC_CMD_DYNAMIC_SENSORS_LIST_OUT_LEN(n_sensors)) {
		WARN_ON(1);
		return -EINVAL;
	}
	spin_lock(&hwmon->update_lock);
	efct_mcdi_remove_dynamic_sensors(efct);
	hwmon->n_dynamic_sensors = n_sensors;
	hwmon->sensor_list = kcalloc(n_sensors, sizeof(struct efct_dynamic_sensor_description),
				     GFP_ATOMIC);
	if (!hwmon->sensor_list) {
		spin_unlock(&hwmon->update_lock);
		return -ENOMEM;
	}

	for (i = 0; i < n_sensors; i++) {
		u32 handle;

		handle = MCDI_ARRAY_DWORD(outbuf,
					  DYNAMIC_SENSORS_LIST_OUT_HANDLES, i);
		efct_mcdi_add_dynamic_sensor(efct, handle, i);
	}

	spin_unlock(&hwmon->update_lock);

	return 0;
}
#endif

static void efct_mcdi_handle_dynamic_sensor_state_change(struct efct_nic *efct, efct_qword_t *ev)
{
	int cont = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_CONT);
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	struct efct_dynamic_sensor_description *sensor;
	efct_dword_t *sensor_reading_entry;
	const char *state_txt;
	int state, value;
	u32 handle;

	spin_lock(&hwmon->update_lock);
	/* Need to receive event with cont = 1 first */
	if (cont) {
		handle = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_DATA);
		sensor = efct_mcdi_get_dynamic_sensor(efct, handle);
		if (!sensor) {
			spin_unlock(&hwmon->update_lock);
			return;
		}

		sensor_reading_entry =
			MCDI_ARRAY_STRUCT_PTR(((efct_dword_t *)hwmon->dma_buf.addr),
					      DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES,
					      sensor->idx);
		/* save sensor index to handle 2nd event of
		 * DYNAMIC_SENSOR_STATE_CHANGE
		 */
		hwmon->pend_sensor_state_handle = handle;
		EFCT_WARN_ON_PARANOID(hwmon->pend_sensor_state_handle <= 0);
		state = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_SRC);
		EFCT_WARN_ON_PARANOID(state >= ARRAY_SIZE(sensor_status_names));
		state_txt = sensor_status_names[state];
		netif_warn(efct, drv, efct->net_dev,
			   "%s: sensor %s state changed to %s\n", efct->name,
			   sensor->name, state_txt);

		MCDI_SET_DWORD(sensor_reading_entry,
			       DYNAMIC_SENSORS_READING_HANDLE,
			       handle);
		MCDI_SET_DWORD(sensor_reading_entry,
			       DYNAMIC_SENSORS_READING_STATE, state);
	} else {
		EFCT_WARN_ON_PARANOID(hwmon->pend_sensor_state_handle < 0);
		handle = hwmon->pend_sensor_state_handle;
		hwmon->pend_sensor_state_handle = -1;

		sensor = efct_mcdi_get_dynamic_sensor(efct, handle);
		if (!sensor) {
			spin_unlock(&hwmon->update_lock);
			return;
		}

		sensor_reading_entry =
			MCDI_ARRAY_STRUCT_PTR(((efct_dword_t *)hwmon->dma_buf.addr),
					      DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES,
					      sensor->idx);
		value = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_DATA);
		MCDI_SET_DWORD(sensor_reading_entry,
			       DYNAMIC_SENSORS_READING_VALUE, value);
	}
	spin_unlock(&hwmon->update_lock);
}

void efct_mcdi_dynamic_sensor_event(struct efct_nic *efct, efct_qword_t *ev)
{
	int code = EFCT_QWORD_FIELD(*ev, MCDI_EVENT_CODE);

	switch (code) {
	case MCDI_EVENT_CODE_DYNAMIC_SENSORS_STATE_CHANGE:
		efct_mcdi_handle_dynamic_sensor_state_change(efct, ev);
		return;
	case MCDI_EVENT_CODE_DYNAMIC_SENSORS_CHANGE:
		netif_info(efct, drv, efct->net_dev,
			   "CODE_DYNAMIC_SENSORS_CHANGE event unsupported\n");
	}
}

static int
efct_mcdi_dynamic_sensor_list_reading_update(struct efct_nic *efct,
					     u32 *handle_list,
					     u32 num_handles)
{
	MCDI_DECLARE_BUF(outbuf,
			 MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_LEN(EFCT_DYNAMIC_SENSOR_READING_UPDATE_MAX_HANDLES));
	MCDI_DECLARE_BUF(inbuf,
			 MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_LEN(EFCT_DYNAMIC_SENSOR_READING_UPDATE_MAX_HANDLES));
	struct efct_dynamic_sensor_description *sensor = NULL;
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	efct_dword_t *sensor_reading_entry;
	efct_dword_t *sensor_out_entry;
	size_t outlen;
	int i, rc = 0;
	u32 handle;

	for (i = 0; i < num_handles; i++)
		MCDI_SET_ARRAY_DWORD(inbuf,
				     DYNAMIC_SENSORS_GET_READINGS_IN_HANDLES, i,
				     handle_list[i]);
	rc = efct_mcdi_rpc(efct, MC_CMD_DYNAMIC_SENSORS_GET_READINGS, inbuf,
			   MC_CMD_DYNAMIC_SENSORS_GET_READINGS_IN_LEN(i),
			  outbuf, sizeof(outbuf), &outlen);
	if (rc)
		return rc;
	/* outlen determins the number of handles returned if any handles
	 * are dropped by FW
	 */
	if (outlen % MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_LEN) {
		WARN_ON(1);
		return -EINVAL;
	}
	i = 0;
	spin_lock(&hwmon->update_lock);
	while (outlen) {
		sensor_out_entry =
			MCDI_ARRAY_STRUCT_PTR(outbuf,
					      DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES,
					      i);
		handle = MCDI_DWORD(sensor_out_entry,
				    DYNAMIC_SENSORS_READING_HANDLE);
		sensor = efct_mcdi_get_dynamic_sensor(efct, handle);
		/* check if sensor was dropped */
		if (IS_ERR(sensor)) {
			i++;
			outlen -= MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_LEN;
			continue;
		}
		sensor_reading_entry =
			MCDI_ARRAY_STRUCT_PTR(((efct_dword_t *)hwmon->dma_buf.addr),
					      DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES,
					      sensor->idx);
		MCDI_SET_DWORD(sensor_reading_entry,
			       DYNAMIC_SENSORS_READING_HANDLE, handle);
		MCDI_SET_DWORD(sensor_reading_entry,
			       DYNAMIC_SENSORS_READING_VALUE,
			       (MCDI_DWORD(sensor_out_entry,
					  DYNAMIC_SENSORS_READING_VALUE)));
		MCDI_SET_DWORD(sensor_reading_entry,
			       DYNAMIC_SENSORS_READING_STATE,
			       (MCDI_DWORD(sensor_out_entry,
					  DYNAMIC_SENSORS_READING_STATE)));

		i++;
		outlen -= MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_LEN;
	}
	spin_unlock(&hwmon->update_lock);
	if (rc == 0)
		hwmon->last_update = jiffies;

	return rc;
}

static int efct_mcdi_dynamic_sensor_reading_update(struct efct_nic *efct)
{
	/* limiting maximum sensors per read to 4 to avoid
	 * -Werror=frame-larger-than=]
	 */
	u32 handles[EFCT_DYNAMIC_SENSOR_INFO_READ_MAX_HANDLES];
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	struct efct_dynamic_sensor_description *sensor;
	u32 j, i = 0;
	int rc = 0;

	sensor = (struct efct_dynamic_sensor_description *)hwmon->sensor_list;
	for (j = 0; j < hwmon->n_dynamic_sensors; j++) {
		handles[i] = sensor[j].handle;
		i++;
		if (i == EFCT_DYNAMIC_SENSOR_INFO_READ_MAX_HANDLES ||
		    (j == (hwmon->n_dynamic_sensors - 1))) {
			rc = efct_mcdi_dynamic_sensor_list_reading_update(efct,
									  handles,
									 i);
			if (!rc)
				i = 0;
			else
				break;
		}
	}

	return rc;
}

static int efct_mcdi_read_dynamic_sensor_list_info(struct efct_nic *efct,
						   u32 *handle_list,
						  u32 num_handles)
{
	MCDI_DECLARE_BUF(outbuf, MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_LEN
			 (EFCT_DYNAMIC_SENSOR_INFO_READ_MAX_HANDLES));
	MCDI_DECLARE_BUF(inbuf, MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_LEN
			 (EFCT_DYNAMIC_SENSOR_INFO_READ_MAX_HANDLES));
	struct efct_dynamic_sensor_description *sensor = NULL;
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	int i, rc = 0;
	size_t outlen;
	u32 handle;

	for (i = 0; i < num_handles; i++) {
		MCDI_SET_ARRAY_DWORD(inbuf, DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_HANDLES,
				     i, handle_list[i]);
	}
	rc = efct_mcdi_rpc(efct, MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS, inbuf,
			   MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_IN_LEN(i),
			  outbuf, sizeof(outbuf), &outlen);
	if (rc)
		return rc;
	i = 0;
	/* outlen determins the number of handles returned if any handles
	 * are dropped by FW
	 */
	if (outlen % MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_LEN) {
		WARN_ON(1);
		return -EINVAL;
	}

	spin_lock(&hwmon->update_lock);
	while (outlen) {
		efct_dword_t *str_ptr =
				MCDI_ARRAY_STRUCT_PTR(outbuf,
						      DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS,
						      i);

		handle = MCDI_DWORD(str_ptr, DYNAMIC_SENSORS_DESCRIPTION_HANDLE);
		sensor = efct_mcdi_get_dynamic_sensor(efct, handle);
		/* check if sensor was dropped */
		if (IS_ERR(sensor)) {
			i++;
			outlen -= MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_LEN;
			continue;
		}
		memcpy(sensor->name,
		       MCDI_PTR(str_ptr, DYNAMIC_SENSORS_DESCRIPTION_NAME),
		       MC_CMD_DYNAMIC_SENSORS_DESCRIPTION_NAME_LEN);
		sensor->type = MCDI_DWORD(str_ptr, DYNAMIC_SENSORS_DESCRIPTION_TYPE);

		sensor->limits[EFCT_SENSOR_LIMIT_WARNING_LO] =
			MCDI_DWORD(str_ptr, DYN_SENSOR_LIMIT_LO_WARNING);
		efct_mcdi_mon_add_attr(efct, sensor->idx,
				       sensor->type,
				      sensor->limits[EFCT_SENSOR_LIMIT_WARNING_LO],
				      hwmon->type_idx[sensor->type],
				      EFCT_HWMON_WARNING_LOW);

		sensor->limits[EFCT_SENSOR_LIMIT_WARNING_HI] =
			MCDI_DWORD(str_ptr, DYN_SENSOR_LIMIT_HI_WARNING);
		efct_mcdi_mon_add_attr(efct, sensor->idx,
				       sensor->type,
				      sensor->limits[EFCT_SENSOR_LIMIT_WARNING_HI],
				      hwmon->type_idx[sensor->type],
				      EFCT_HWMON_WARNING_HIGH);

		sensor->limits[EFCT_SENSOR_LIMIT_CRITICAL_LO] =
			MCDI_DWORD(str_ptr, DYN_SENSOR_LIMIT_LO_CRITICAL);
		if (sensor->limits[EFCT_SENSOR_LIMIT_CRITICAL_LO] != 0)
			efct_mcdi_mon_add_attr(efct, sensor->idx,
					       sensor->type,
					      sensor->limits[EFCT_SENSOR_LIMIT_CRITICAL_LO],
					      hwmon->type_idx[sensor->type],
					      EFCT_HWMON_CRIT_LOW);

		sensor->limits[EFCT_SENSOR_LIMIT_CRITICAL_HI] =
			MCDI_DWORD(str_ptr, DYN_SENSOR_LIMIT_HI_CRITICAL);
		if (sensor->limits[EFCT_SENSOR_LIMIT_CRITICAL_HI] != 0)
			efct_mcdi_mon_add_attr(efct, sensor->idx,
					       sensor->type,
					      sensor->limits[EFCT_SENSOR_LIMIT_CRITICAL_HI],
					      hwmon->type_idx[sensor->type],
					      EFCT_HWMON_CRIT_HIGH);

		efct_mcdi_mon_add_attr(efct, sensor->idx, sensor->type, 0,
				       hwmon->type_idx[sensor->type], EFCT_HWMON_LABEL);
		efct_mcdi_mon_add_attr(efct, sensor->idx, sensor->type, 0,
				       hwmon->type_idx[sensor->type], EFCT_HWMON_INPUT);
		hwmon->type_idx[sensor->type]++;
		i++;
		outlen -= MC_CMD_DYNAMIC_SENSORS_GET_DESCRIPTIONS_OUT_SENSORS_LEN;
	}
	spin_unlock(&hwmon->update_lock);

	return 0;
}

/* This function is assumed to be called only after @has_dynamic_sensors
 * has returned true or DYNAMIC_SENSOR_LIST event received
 */
static int efct_mcdi_read_dynamic_sensor_info(struct efct_nic *efct)
{
	/* limiting maximum sensors per read to 4 to avoid
	 * -Werror=frame-larger-than=]
	 */
	u32 handles[EFCT_DYNAMIC_SENSOR_INFO_READ_MAX_HANDLES];
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	struct efct_dynamic_sensor_description *sensor;
	u32 j, i = 0;
	int rc = 0;

	sensor = (struct efct_dynamic_sensor_description *)hwmon->sensor_list;
	for (j = 0; j < hwmon->n_dynamic_sensors; j++) {
		handles[i] = sensor[j].handle;
		i++;
		if (i == EFCT_DYNAMIC_SENSOR_INFO_READ_MAX_HANDLES ||
		    j == (hwmon->n_dynamic_sensors - 1)) {
			rc = efct_mcdi_read_dynamic_sensor_list_info(efct,
								     handles, i);
			if (!rc)
				i = 0;
			else
				break;
		}
	}

	return rc;
}

static int efct_mcdi_mon_get_dynamic_entry(struct efct_nic *efct,
					   u32 index,
					   efct_dword_t **entry)
{
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	int rc;

	BUILD_BUG_ON(MC_CMD_READ_SENSORS_OUT_LEN != 0);

	/* Use cached value if last update was < 1 s ago */
	if (time_before(jiffies, hwmon->last_update + HZ))
		rc = 0;
	else
		rc = efct_mcdi_dynamic_sensor_reading_update(efct);

	/* Copy out the requested entry. Entries for dynamic sensors are
	 * commposed by three efct_dword_t values: handle, value, state.
	 */
	spin_lock(&hwmon->update_lock);
	*entry = &((efct_dword_t *)hwmon->dma_buf.addr)[index * 3];
	spin_unlock(&hwmon->update_lock);

	return rc;
}

static int efct_mcdi_mon_get_value(struct efct_nic *efct, u32 index,
				   enum hwmon_sensor_types hwmon_type,
				   u32 *value)
{
	efct_dword_t *dyn_entry;
	u32 state;
	int rc;

	rc = efct_mcdi_mon_get_dynamic_entry(efct, index, &dyn_entry);
	if (rc)
		return rc;

	state = MCDI_DWORD(dyn_entry, DYNAMIC_SENSORS_READING_STATE);
	if (state == MC_CMD_DYNAMIC_SENSORS_READING_NO_READING)
		return -EBUSY;

	*value = MCDI_DWORD(dyn_entry, DYNAMIC_SENSORS_READING_VALUE);

	switch (hwmon_type) {
	case hwmon_temp:
		/* Convert temperature from degrees to milli-degrees Celsius */
		*value *= 1000;
		break;
	case hwmon_power:
		/* Convert power from watts to microwatts */
		*value *= 1000000;
		break;
	default:
		/* No conversion needed */
		break;
	}
	return 0;
}

static u32 efct_mcdi_mon_get_limit(struct efct_mcdi_mon_attribute *mon_attr)
{
	u32 value;

	value = mon_attr->limit_value;

	switch (mon_attr->hwmon_type) {
	case hwmon_temp:
		/* Convert temperature from degrees to milli-degrees Celsius */
		value *= 1000;
		break;
	case hwmon_power:
		/* Convert power from watts to microwatts */
		value *= 1000000;
		break;
	default:
		/* No conversion needed */
		break;
	}
	return value;
}

static int efct_mcdi_mon_get_state(struct efct_nic *efct, u32 index,
				   u32 *value)
{
	efct_dword_t *dyn_entry;
	int rc = -1;

	rc = efct_mcdi_mon_get_dynamic_entry(efct, index, &dyn_entry);
	if (rc)
		return rc;
	rc = MCDI_DWORD(dyn_entry, DYNAMIC_SENSORS_READING_STATE);

	return rc;
}

#if defined(EFCT_USE_KCOMPAT) && !defined(EFCT_HAVE_HWMON_DEVICE_REGISTER_WITH_INFO)
static ssize_t efct_mcdi_mon_show(struct kobject *kobj, struct attribute *attr, char *buf)
{
	struct efct_mcdi_mon_attribute *mon_attr =
		container_of(attr, struct efct_mcdi_mon_attribute, dev_attr);
	struct efct_mcdi_mon *hwmon = container_of(kobj, struct efct_mcdi_mon, kobj);
	struct efct_dynamic_sensor_description *sensor;
	u32 value = 0;
	int state = 0;
	int rc;

	switch (mon_attr->hwmon_attribute) {
	case EFCT_HWMON_INPUT:
		rc = efct_mcdi_mon_get_value(hwmon->efct, mon_attr->index,
					     mon_attr->hwmon_type, &value);
		if (rc)
			return rc;
		return scnprintf(buf, PAGE_SIZE, "%u\n", value);
	case EFCT_HWMON_WARNING_LOW:
	case EFCT_HWMON_WARNING_HIGH:
	case EFCT_HWMON_CRIT_LOW:
	case EFCT_HWMON_CRIT_HIGH:
		value = efct_mcdi_mon_get_limit(mon_attr);
		BUILD_BUG_ON(MC_CMD_READ_SENSORS_OUT_LEN != 0);
		return scnprintf(buf, PAGE_SIZE, "%u\n", value);
	case EFCT_HWMON_ALARM:
		rc = efct_mcdi_mon_get_state(hwmon->efct, mon_attr->index, &state);
		if (rc)
			return rc;
		return scnprintf(buf, PAGE_SIZE,
				 "%d\n", state != MC_CMD_DYNAMIC_SENSORS_READING_OK);
	case EFCT_HWMON_LABEL:
		sensor = efct_mcdi_get_dynamic_sensor(hwmon->efct, mon_attr->index + 1);
		return scnprintf(buf, PAGE_SIZE, "%s\n", sensor->name);
	case EFCT_HWMON_NAME:
		return scnprintf(buf, PAGE_SIZE, "%s\n", KBUILD_MODNAME);
	}

	return 0;
}

static int efct_mcdi_mon_create_files(struct device *dev,
				      struct efct_mcdi_mon *hwmon)
{
	static struct attribute **sysfs_sensor_attributes;
	int rc = 0, i;

	sysfs_sensor_attributes = kcalloc(hwmon->n_attrs + 1,
					  sizeof(struct attribute *), GFP_KERNEL);
	if (!sysfs_sensor_attributes)
		return -ENOMEM;

	for (i = 0; i < hwmon->n_attrs; i++) {
		struct efct_mcdi_mon_attribute *attr = &hwmon->attrs[i];
		const char *hwmon_prefix;

		sysfs_attr_init(&attr->dev_attr.attr);
		attr->dev_attr.mode = 0444;

		switch (attr->hwmon_type) {
		case hwmon_temp:
			hwmon_prefix = "temp";
			break;
		case hwmon_fan:
			/* This is likely to be a heatsink, but there
			 * is no convention for representing cooling
			 * devices other than fans.
			 */
			hwmon_prefix = "fan";
			break;
		case hwmon_in:
			hwmon_prefix = "in";
			break;
		case hwmon_curr:
			hwmon_prefix = "curr";
			break;
		case hwmon_power:
			hwmon_prefix = "power";
			break;
		case hwmon_chip:
			continue;
		default:
			dev_warn(dev, "Unknown HW monitor type %d\n",
				 attr->hwmon_type);
			continue;
		}

		switch (attr->hwmon_attribute) {
		case EFCT_HWMON_INPUT:
			snprintf(attr->name, sizeof(attr->name), "%s%hhu_input",
				 hwmon_prefix, attr->file_index);
			break;
		case EFCT_HWMON_WARNING_LOW:
			snprintf(attr->name, sizeof(attr->name), "%s%hhu_min",
				 hwmon_prefix, attr->file_index);
			break;
		case EFCT_HWMON_WARNING_HIGH:
			snprintf(attr->name, sizeof(attr->name), "%s%hhu_max",
				 hwmon_prefix, attr->file_index);
			break;
		case EFCT_HWMON_CRIT_LOW:
			snprintf(attr->name, sizeof(attr->name), "%s%hhu_crit_low",
				 hwmon_prefix, attr->file_index);
			break;
		case EFCT_HWMON_CRIT_HIGH:
			snprintf(attr->name, sizeof(attr->name), "%s%hhu_crit_high",
				 hwmon_prefix, attr->file_index);
			break;
		case EFCT_HWMON_ALARM:
			snprintf(attr->name, sizeof(attr->name), "%s%hhu_alarm",
				 hwmon_prefix, attr->file_index);
			break;
		case EFCT_HWMON_LABEL:
			snprintf(attr->name, sizeof(attr->name), "%s%hhu_label",
				 hwmon_prefix, attr->file_index);
			break;
		case EFCT_HWMON_NAME:
			snprintf(attr->name, sizeof(attr->name), "name");
			break;
		default:
			dev_warn(dev, "Unknown HW monitor attribute %d\n",
				 attr->hwmon_attribute);
			continue;
		}

		attr->sensorops.show = efct_mcdi_mon_show;
		attr->dev_attr.name = attr->name;
		hwmon->sensor_ktype.sysfs_ops = &attr->sensorops;
		sysfs_sensor_attributes[i] = &attr->dev_attr;
	}

	hwmon->sysfs_sensor_attr_group.attrs = sysfs_sensor_attributes;
	rc = kobject_init_and_add(&hwmon->kobj, &hwmon->sensor_ktype, &dev->kobj,
				  "port%d", hwmon->efct->port_num);
	if (rc) {
		pr_err("Not able to create directory inside sysfs\n");
		kfree(sysfs_sensor_attributes);
		return rc;
	}

	if (sysfs_create_group(&hwmon->kobj, &hwmon->sysfs_sensor_attr_group)) {
		pr_err("Not able to add files. sysfs_create_group failed\n");
		kobject_put(&hwmon->kobj);
		kfree(sysfs_sensor_attributes);
		return -1;
	}

	return rc;
}

static void efct_mcdi_mon_remove_files(struct device *dev,
				       struct efct_mcdi_mon *hwmon)
{
	sysfs_remove_group(&hwmon->kobj, &hwmon->sysfs_sensor_attr_group);
	kobject_put(&hwmon->kobj);
}

#else
/* These defines must match with the efct_attrib_map array below. */
#define EFCT_HWMON_TEMP_CONFIG	(HWMON_T_INPUT | HWMON_T_MIN | HWMON_T_MAX | \
				HWMON_T_LCRIT | HWMON_T_CRIT | HWMON_T_ALARM | HWMON_T_LABEL)
#define EFCT_HWMON_IN_CONFIG	(HWMON_I_INPUT | HWMON_I_MIN | HWMON_I_MAX | \
				 HWMON_I_LCRIT | HWMON_I_CRIT | HWMON_I_ALARM | HWMON_I_LABEL)
#define EFCT_HWMON_CURR_CONFIG	(HWMON_C_INPUT | HWMON_C_MIN | HWMON_C_MAX | \
				 HWMON_C_LCRIT | HWMON_C_CRIT | HWMON_C_ALARM | HWMON_C_LABEL)
#define EFCT_HWMON_POWER_CONFIG	(HWMON_P_INPUT | HWMON_P_LABEL)
#define EFCT_HWMON_FAN_CONFIG	(HWMON_F_ALARM | HWMON_F_LABEL)

static const u32
efct_attrib_map[EFCT_HWMON_TYPES_COUNT][EFCT_HWMON_ATTRIBUTE_COUNT] = {
	[hwmon_temp] = { HWMON_T_INPUT, HWMON_T_MIN, HWMON_T_MAX, HWMON_T_LCRIT, HWMON_T_CRIT,
			 HWMON_T_ALARM, HWMON_T_LABEL, 0 },
	[hwmon_in] = { HWMON_I_INPUT, HWMON_I_MIN, HWMON_I_MAX, HWMON_I_LCRIT, HWMON_I_CRIT,
			 HWMON_I_ALARM, HWMON_I_LABEL, 0 },
	[hwmon_curr] = { HWMON_C_INPUT, HWMON_C_MIN, HWMON_C_MAX, HWMON_C_LCRIT, HWMON_C_CRIT,
			 HWMON_C_ALARM, HWMON_C_LABEL, 0 },
	[hwmon_power] = { HWMON_P_INPUT, 0, 0, 0, 0, 0, HWMON_P_LABEL, 0 },
	[hwmon_fan] = { 0, 0, 0, 0, 0, HWMON_F_ALARM, HWMON_F_LABEL, 0},
};

static int efct_mcdi_mon_create_files(struct device *dev,
				      struct efct_mcdi_mon *hwmon)
{
	return 0;
}

static void efct_mcdi_mon_remove_files(struct device *dev,
				       struct efct_mcdi_mon *hwmon)
{
}

static struct efct_mcdi_mon_attribute *
efct_hwmon_get_attribute(const struct efct_nic *efct,
			 enum hwmon_sensor_types type, u32 attr, int channel)
{
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon((struct efct_nic *)efct);
	struct efct_mcdi_mon_attribute *attribute;
	enum efct_hwmon_attribute hwmon_attribute;
	int i;

	if (type > EFCT_HWMON_TYPES_COUNT)
		return NULL;
	if (!hwmon || !hwmon->n_attrs) {
		pr_err("Error: Seems stale sysfs read, Can be X3 driver bug\n");
		return NULL;
	}
	for (i = 0; i < EFCT_HWMON_ATTRIBUTE_COUNT; i++) {
		if (efct_attrib_map[type][i] == BIT(attr)) {
			hwmon_attribute = i;
			break;
		}
	}
	if (i ==  EFCT_HWMON_ATTRIBUTE_COUNT)
		return NULL;

	for (i = 0; i < hwmon->n_attrs; i++) {
		attribute = &hwmon->attrs[i];
		if (attribute->hwmon_type == type &&
		    attribute->hwmon_attribute == hwmon_attribute &&
		    attribute->file_index == channel)
			return attribute;
	}
	return NULL;
}

static int efct_hwmon_read(struct device *dev,
			   enum hwmon_sensor_types type,
			  u32 attr, int channel, long *val)
{
	struct efct_nic *efct = dev_get_drvdata(dev);
	struct efct_mcdi_mon_attribute *mon_attr;
	u32 value = 0;
	int rc;

	*val = 0;
	mon_attr = efct_hwmon_get_attribute(efct, type, attr, channel);
	if (!mon_attr)
		return -EOPNOTSUPP;

	switch (mon_attr->hwmon_attribute) {
	case EFCT_HWMON_INPUT:
		rc = efct_mcdi_mon_get_value(efct, mon_attr->index,
					     mon_attr->hwmon_type, &value);
		if (rc < 0)
			return rc;
		break;
	case EFCT_HWMON_WARNING_LOW:
	case EFCT_HWMON_WARNING_HIGH:
	case EFCT_HWMON_CRIT_LOW:
	case EFCT_HWMON_CRIT_HIGH:
		value = efct_mcdi_mon_get_limit(mon_attr);
		break;
	case EFCT_HWMON_ALARM:
		rc = efct_mcdi_mon_get_state(efct, mon_attr->index, &value);
		if (rc)
			return rc;
		value = (value != MC_CMD_DYNAMIC_SENSORS_READING_OK);
		break;
#ifndef EFCT_HAVE_HWMON_READ_STRING
	case EFCT_HWMON_LABEL:
		return -EOPNOTSUPP;
#endif
	default:
		WARN_ONCE(1, "Unhandled HW sensor read\n");
		return -EOPNOTSUPP;
	}
	*val = value;
	return 0;
}

#ifdef EFCT_HAVE_HWMON_READ_STRING
static int efct_hwmon_read_string(struct device *dev,
				  enum hwmon_sensor_types type,
				 u32 attr, int channel,
#ifdef EFCT_HAVE_HWMON_READ_STRING_CONST
				 const char **str
#else
				 char **str
#endif
				)
{
	const struct efct_nic *efct = dev_get_drvdata(dev);
	struct efct_nic *efct_temp = dev_get_drvdata(dev);
	struct efct_dynamic_sensor_description *sensor;
	struct efct_mcdi_mon_attribute *mon_attr;

	mon_attr = efct_hwmon_get_attribute(efct, type, attr, channel);
	if (!mon_attr)
		return 1;

	sensor = efct_mcdi_get_dynamic_sensor(efct_temp, mon_attr->index + 1);
	if (!sensor) {
		WARN(1, "%s: sensor not found\n", __func__);
		*str = NULL;
	} else {
		*str = sensor->name;
	}

	return 0;
}
#endif

static umode_t efct_hwmon_is_visible(const void *data,
				     enum hwmon_sensor_types type,
				    u32 attr, int channel)
{
	const struct efct_nic *efct = data;
	struct efct_mcdi_mon_attribute *mon_attr;

	mon_attr = efct_hwmon_get_attribute(efct, type, attr, channel);
	if (mon_attr)
		return 0444;
	else
		return 0;
}

static const u32 efct_temp_config[] = {
	EFCT_HWMON_TEMP_CONFIG,
	EFCT_HWMON_TEMP_CONFIG,
	EFCT_HWMON_TEMP_CONFIG,
	EFCT_HWMON_TEMP_CONFIG,
	EFCT_HWMON_TEMP_CONFIG,
	0
};

static const struct hwmon_channel_info efct_temp = {
	.type = hwmon_temp,
	.config = efct_temp_config,
};

static const u32 efct_in_config[] = {
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	EFCT_HWMON_IN_CONFIG,
	0
};

static const struct hwmon_channel_info efct_in = {
	.type = hwmon_in,
	.config = efct_in_config,
};

static const u32 efct_curr_config[] = {
	EFCT_HWMON_CURR_CONFIG,
	EFCT_HWMON_CURR_CONFIG,
	EFCT_HWMON_CURR_CONFIG,
	EFCT_HWMON_CURR_CONFIG,
	0
};

static const struct hwmon_channel_info efct_curr = {
	.type = hwmon_curr,
	.config = efct_curr_config,
};

static const u32 efct_power_config[] = {
	EFCT_HWMON_POWER_CONFIG,
	EFCT_HWMON_POWER_CONFIG,
	EFCT_HWMON_POWER_CONFIG,
	EFCT_HWMON_POWER_CONFIG,
	EFCT_HWMON_POWER_CONFIG,
	0
};

static const struct hwmon_channel_info efct_power = {
	.type = hwmon_power,
	.config = efct_power_config,
};

static const u32 efct_fan_config[] = {
	EFCT_HWMON_FAN_CONFIG,
	0
};

static const struct hwmon_channel_info efct_fan = {
	.type = hwmon_fan,
	.config = efct_fan_config,
};

static const struct hwmon_channel_info *efct_hwmon_info[] = {
	&efct_temp,
	&efct_in,
	&efct_curr,
	&efct_power,
	&efct_fan,
	NULL
};

static const struct hwmon_ops efct_hwmon_ops = {
	.is_visible = efct_hwmon_is_visible,
	.read = efct_hwmon_read,
#if defined(EFCT_HAVE_HWMON_READ_STRING)
	.read_string = efct_hwmon_read_string,
#endif
};
#endif

static int efct_mcdi_hwmon_probe(struct efct_nic *efct, u32 n_sensors)
{
	u32 sensor_entry_len = MC_CMD_DYNAMIC_SENSORS_GET_READINGS_OUT_VALUES_LEN;
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	int rc, i = 0;
	u32 n_attrs;

	hwmon->chip_info = NULL;

	rc = efct_nic_alloc_buffer(efct, &hwmon->dma_buf,
				   n_sensors * sensor_entry_len, GFP_KERNEL);
	if (rc)
		return rc;

	/* Allocate space for the maximum possible number of
	 * attributes for this set of sensors: name of the driver plus
	 * value, min, max, crit, alarm and label for each sensor.
	 */
	n_attrs = 1 + 6 * n_sensors;
	hwmon->attrs = kcalloc(n_attrs, sizeof(*hwmon->attrs), GFP_KERNEL);
	if (!hwmon->attrs)
		return -ENOMEM;

	efct_mcdi_mon_add_attr(efct, 0, 0, 0, 0, EFCT_HWMON_NAME);

	/* Initializing the attr counter to 0 */
	for (i = 0; i < EFCT_HWMON_TYPES_COUNT; i++)
		hwmon->type_idx[i] = 0;

	rc = efct_mcdi_read_dynamic_sensor_info(efct);
	if (rc)
		return rc;

#if defined(EFCT_USE_KCOMPAT) && defined(EFCT_HAVE_HWMON_DEVICE_REGISTER_WITH_INFO)
	hwmon->chip_info = kzalloc(sizeof(*hwmon->chip_info), GFP_KERNEL);
	if (!hwmon->chip_info)
		return -ENOMEM;

	hwmon->chip_info->ops = &efct_hwmon_ops;
	hwmon->chip_info->info = efct_hwmon_info;
#endif

	hwmon->device = hwmon_device_register_with_info(&efct->efct_dev->pci_dev->dev,
							efct->name, efct,
							hwmon->chip_info,
							NULL);
	if (IS_ERR(hwmon->device)) {
		kfree(hwmon->chip_info);
		hwmon->chip_info = NULL;
		return PTR_ERR(hwmon->device);
	}

	rc = efct_mcdi_mon_create_files(&efct->efct_dev->pci_dev->dev, hwmon);

	return rc;
}

int efct_mcdi_mon_probe(struct efct_nic *efct)
{
	bool has_dynamic_sensors = efct_nic_has_dynamic_sensors(efct);
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);
	u32 n_sensors;
	int rc = -1;

	hwmon->efct = efct;
	/* Do not probe twice */
	if (hwmon->dma_buf.addr)
		return 0;

	if (!has_dynamic_sensors) {
		netif_err(efct, hw, efct->net_dev,
			  "Expected dynamic sensor feature not supported by FW\n");
		return rc;
	}

	/* Find out how many sensors are present */
	n_sensors = 0;

	spin_lock_init(&hwmon->update_lock);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_RHASHTABLE_LOOKUP_FAST)
	rhashtable_init(&hwmon->sensor_table, &sensor_entry_params);
	rc = efct_mcdi_read_dynamic_sensor_list(efct);
	n_sensors = hwmon->n_dynamic_sensors;
#else
	netif_err(efct, hw, efct->net_dev,
		  "Kernel is too old to support dynamic sensors\n");
	rc = -ENOTSUPP;
#endif

	if (rc)
		return rc;

	if (!n_sensors)
		return 0;

	rc  = efct_mcdi_hwmon_probe(efct, n_sensors);
	if (rc) {
		efct_mcdi_mon_remove(efct);
		return rc;
	}

	efct_mcdi_dynamic_sensor_reading_update(efct);

	return rc;
}

static void efct_mcdi_hwmon_remove(struct efct_nic *efct)
{
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);

	efct_mcdi_mon_remove_files(&efct->efct_dev->pci_dev->dev, hwmon);
	if (!IS_ERR_OR_NULL(hwmon->device))
		hwmon_device_unregister(hwmon->device);
	kfree(hwmon->attrs);

	kfree(hwmon->chip_info);
	hwmon->chip_info = NULL;

	spin_lock(&hwmon->update_lock);

	hwmon->attrs = NULL;
	hwmon->n_attrs = 0;
	kfree(hwmon->sensor_list);
	hwmon->sensor_list = NULL;
	hwmon->n_dynamic_sensors = 0;

	efct_nic_free_buffer(efct, &hwmon->dma_buf);

	spin_unlock(&hwmon->update_lock);
}

void efct_mcdi_mon_remove(struct efct_nic *efct)
{
	struct efct_mcdi_mon *hwmon = efct_mcdi_mon(efct);

	if (!hwmon || !hwmon->dma_buf.addr)
		return;
	efct_mcdi_hwmon_remove(efct);
#if !defined(EFCT_USE_KCOMPAT) || defined(EFCT_HAVE_RHASHTABLE_LOOKUP_FAST)
	if (efct_nic_has_dynamic_sensors(efct))
		rhashtable_free_and_destroy(&hwmon->sensor_table,
					    NULL,
					    NULL);
#endif
}

#endif /* CONFIG_XILINX_MCDI_MON */
