// SPDX-License-Identifier: GPL-2.0
/****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#include <linux/bitops.h>
#include <linux/delay.h>
#include <linux/interrupt.h>
#include <linux/pci.h>
#include <linux/module.h>
#include <linux/seq_file.h>

#include "net_driver.h"
#include "bitfield.h"
#include "nic.h"
#include "efct_regs.h"
#include "io.h"
#include "mcdi_pcol.h"

/**
 * efct_nic_describe_stats - Describe supported statistics for ethtool
 * @desc: Array of &struct efct_hw_stat_desc describing the statistics
 * @count: Length of the @desc array
 * @mask: Bitmask of which elements of @desc are enabled
 * @names: Buffer to copy names to, or %NULL.  The names are copied
 *	starting at intervals of %ETH_GSTRING_LEN bytes.
 *
 * Returns the number of visible statistics, i.e. the number of set
 * bits in the first @count bits of @mask for which a name is defined.
 */
int efct_nic_describe_stats(const struct efct_hw_stat_desc *desc, size_t count,
			    const unsigned long *mask, u8 *names)
{
	int visible = 0;
	int index;

	for_each_set_bit(index, mask, count) {
		if (desc[index].name) {
			if (names) {
				strscpy(names, desc[index].name, ETH_GSTRING_LEN);
				names += ETH_GSTRING_LEN;
			}
			++visible;
		}
	}

	return visible;
}

/**
 * efct_nic_update_stats - Convert statistics DMA buffer to array of u64
 * @desc: Array of &struct efct_hw_stat_desc describing the DMA buffer
 *	layout.  DMA widths of 0, 16, 32 and 64 are supported; where
 *	the width is specified as 0 the corresponding element of
 *	@stats is not updated.
 * @count: Length of the @desc array
 * @mask: Bitmask of which elements of @desc are enabled
 * @stats: Buffer to update with the converted statistics.  The length
 *	of this array must be at least @count.
 * @mc_initial_stats: Copy of DMA buffer containing initial stats. Subtracted
 *	from the stats in mc_stats.
 * @mc_stats: DMA buffer containing hardware statistics
 */
void efct_nic_update_stats(const struct efct_hw_stat_desc *desc, size_t count,
			   const unsigned long *mask, u64 *stats,
			  const void *mc_initial_stats, const void *mc_stats)
{
	__le64 zero = 0;
	size_t index;

	for_each_set_bit(index, mask, count) {
		if (desc[index].dma_width) {
			const void *addr = mc_stats ? mc_stats + desc[index].offset : &zero;
			const void *init = mc_initial_stats && mc_stats ? mc_initial_stats +
					   desc[index].offset : &zero;

			switch (desc[index].dma_width) {
			case 16:
				stats[index] = le16_to_cpup((__le16 *)addr) -
						le16_to_cpup((__le16 *)init);
				break;
			case 32:
				stats[index] = le32_to_cpup((__le32 *)addr) -
						le32_to_cpup((__le32 *)init);
				break;
			case 64:
				stats[index] = le64_to_cpup((__le64 *)addr) -
						le64_to_cpup((__le64 *)init);
				break;
			default:
				WARN_ON_ONCE(1);
				stats[index] = 0;
				break;
			}
		}
	}
}

int efct_nic_alloc_buffer(struct efct_nic *efct, struct efct_buffer *buffer,
			  u32 len, gfp_t gfp_flags)
{
	buffer->addr = dma_alloc_coherent(&efct->efct_dev->pci_dev->dev, len,
					  &buffer->dma_addr, gfp_flags);
	if (!buffer->addr)
		return -ENOMEM;
	buffer->len = len;
	memset(buffer->addr, 0, len);
	return 0;
}

void efct_nic_free_buffer(struct efct_nic *efct, struct efct_buffer *buffer)
{
	if (buffer->addr) {
		dma_free_coherent(&efct->efct_dev->pci_dev->dev, buffer->len,
				  buffer->addr, buffer->dma_addr);
		buffer->addr = NULL;
	}
}

/* Hook interrupt handler(s)
 */

void efct_set_interrupt_affinity(struct efct_nic *efct)
{
	struct efct_device *efct_dev;
	struct pci_dev *pci_dev;
	u32 cpu;
	int i;

	efct_dev = efct->efct_dev;
	pci_dev = efct->efct_dev->pci_dev;
	for (i = 0; i < efct->max_evq_count; ++i) {
		if (!efct->evq[i].msi.irq)
			continue;
		if (efct_dev->dist_layout == RX_LAYOUT_DISTRIBUTED)
			cpu = cpumask_local_spread(efct->evq[i].msi.idx,
						   pcibus_to_node(pci_dev->bus));
		else
			cpu = cpumask_local_spread(efct_dev->separated_rx_cpu,
						   pcibus_to_node(pci_dev->bus));
		irq_set_affinity_hint(efct->evq[i].msi.irq, cpumask_of(cpu));
		netif_dbg(efct, ifup, efct->net_dev,
			  "EVQ : %u IRQ : %u IDX : %u CPU : %u\n",
			  i, efct->evq[i].msi.irq, efct->evq[i].msi.idx, cpu);
	}
}

void efct_clear_interrupt_affinity(struct efct_nic *efct)
{
	int i;

	for (i = 0; i < efct->max_evq_count; ++i) {
		if (!efct->evq[i].msi.irq)
			continue;
		irq_set_affinity_hint(efct->evq[i].msi.irq, NULL);
	}
}

static u32
efx_device_check_pcie_link(struct pci_dev *pdev, u32 *actual_width,
			   u32 *max_width, u32 *actual_speed,
			   u32 *nic_bandwidth)
{
	int cap = pci_find_capability(pdev, PCI_CAP_ID_EXP);
	u32 nic_speed;
	u16 lnksta;
	u16 lnkcap;

	*actual_speed = 0;
	*actual_width = 0;
	*max_width = 0;
	*nic_bandwidth = 0;

	if (!cap ||
	    pci_read_config_word(pdev, cap + PCI_EXP_LNKSTA, &lnksta) ||
	    pci_read_config_word(pdev, cap + PCI_EXP_LNKCAP, &lnkcap))
		return 0;

	*actual_width = (lnksta & PCI_EXP_LNKSTA_NLW) >>
			__ffs(PCI_EXP_LNKSTA_NLW);

	*max_width = (lnkcap & PCI_EXP_LNKCAP_MLW) >> __ffs(PCI_EXP_LNKCAP_MLW);
	*actual_speed = (lnksta & PCI_EXP_LNKSTA_CLS);

	nic_speed = 1;
	if (lnkcap & PCI_EXP_LNKCAP_SLS_5_0GB)
		nic_speed = 2;
	/* PCIe Gen3 capabilities are in a different config word. */
	if (!pci_read_config_word(pdev, cap + PCI_EXP_LNKCAP2, &lnkcap)) {
		if (lnkcap & PCI_EXP_LNKCAP2_SLS_8_0GB)
			nic_speed = 3;
	}

	*nic_bandwidth = *max_width << (nic_speed - 1);

	return nic_speed;
}

void
efct_nic_check_pcie_link(struct efct_device *efct_dev, u32 desired_bandwidth,
			 u32 *actual_width, u32 *actual_speed)
{
	struct pci_dev *pdev = efct_dev->pci_dev;
	u32 nic_bandwidth;
	u32 bandwidth = 0;
	u32 nic_width = 0;
	u32 nic_speed = 0;
	u32 width = 0;
	u32 speed = 0;

	nic_speed = efx_device_check_pcie_link(pdev, &width, &nic_width, &speed,
					       &nic_bandwidth);

	if (!nic_speed)
		goto out;

	if (width > nic_width)
		pci_dbg(pdev, "PCI Express width is %d, with maximum expected %d. If running on a virtualized platform this is fine, otherwise it indicates a PCI problem.\n",
			width, nic_width);

	bandwidth = width << (speed - 1);

	if (desired_bandwidth > nic_bandwidth)
		/* You can desire all you want, it ain't gonna happen. */
		desired_bandwidth = nic_bandwidth;

	if (desired_bandwidth && bandwidth < desired_bandwidth) {
		pci_warn(pdev,
			 "This Network Adapter requires the equivalent of %d lanes at PCI Express %d speed for full throughput, but is currently limited to %d lanes at PCI Express %d speed.\n",
			 desired_bandwidth > EFX_BW_PCIE_GEN3_X8 ? 16 : 8,
			 nic_speed, width, speed);
		 pci_warn(pdev, "Consult your motherboard documentation to find a more suitable slot\n");
	} else if (bandwidth < nic_bandwidth) {
		pci_warn(pdev,
			 "This Network Adapter requires a slot with %d lanes at PCI Express %d speed for optimal latency, but is currently limited to %d lanes at PCI Express %d speed\n",
			 nic_width, nic_speed, width, speed);
	}

out:
	if (actual_width)
		*actual_width = width;

	if (actual_speed)
		*actual_speed = speed;
}
