/* SPDX-License-Identifier: GPL-2.0
 ****************************************************************************
 * Driver for Xilinx network controllers and boards
 * Copyright 2021 Xilinx Inc.
 *
 * This program is free software; you can redistribute it and/or modify it
 * under the terms of the GNU General Public License version 2 as published
 * by the Free Software Foundation, incorporated herein by reference.
 */

#ifndef NIC_H
#define NIC_H

#ifndef EFCT_USE_KCOMPAT
#include <linux/net_tstamp.h>
#endif
#include <linux/timer.h>
#include "net_driver.h"
#include "mcdi.h"
#include "mcdi_port_common.h"

/* PCIe link bandwidth measure:
 * bw = (width << (speed - 1))
 */
#define EFX_BW_PCIE_GEN1_X8  (8  << (1 - 1))
#define EFX_BW_PCIE_GEN2_X8  (8  << (2 - 1))
#define EFX_BW_PCIE_GEN3_X8  (8  << (3 - 1))
#define EFX_BW_PCIE_GEN3_X16 (16 << (3 - 1))

enum {
	EFCT_REV = 6,
};

#define EFCT_MC_STATS_GENERATION_INVALID ((__force __le64)(-1))
/* NIC-generic software stats */
enum {
	GENERIC_STAT_rx_drops,
	GENERIC_STAT_tx_drops,
	GENERIC_STAT_COUNT
};

#define EFCT_GENERIC_SW_STAT(ext_name)                          \
	[GENERIC_STAT_ ## ext_name] = { #ext_name, 0, 0 }

int efct_nic_copy_stats(struct efct_nic *efct, __le64 *dest);
int efct_nic_describe_stats(const struct efct_hw_stat_desc *desc, size_t count,
			    const unsigned long *mask, u8 *names);
void efct_nic_update_stats(const struct efct_hw_stat_desc *desc, size_t count,
			   const unsigned long *mask, u64 *stats,
			   const void *mc_initial_stats, const void *mc_stats);

static inline int efct_nic_rev(struct efct_nic *efct)
{
	return efct->type->revision;
}

static inline bool efct_nic_hw_unavailable(struct efct_nic *efct)
{
	if (efct->type->hw_unavailable)
		return efct->type->hw_unavailable(efct);
	return false;
}

static inline bool efct_nic_mcdi_ev_pending(struct efct_nic *efct, u16 index)
{
	return efct->type->ev_mcdi_pending(efct, index);
}

static inline bool efct_nic_has_dynamic_sensors(struct efct_nic *efct)
{
	if (efct->type->has_dynamic_sensors)
		return efct->type->has_dynamic_sensors(efct);

	return false;
}

static inline int efct_nic_reset_stats(struct efct_nic *efct)
{
	return efct_mcdi_mac_stats(efct, EFCT_STATS_NODMA, 0, efct->mc_initial_stats,
				  efct->num_mac_stats * sizeof(__le64));
}

/* Global Resources */
int efct_nic_alloc_buffer(struct efct_nic *efct, struct efct_buffer *buffer,
			  u32 len, gfp_t gfp_flags);
void efct_nic_free_buffer(struct efct_nic *efct, struct efct_buffer *buffer);
void efct_set_interrupt_affinity(struct efct_nic *efct);
void efct_clear_interrupt_affinity(struct efct_nic *efct);
void efct_nic_check_pcie_link(struct efct_device *efct_dev, u32 desired_bandwidth,
			      u32 *actual_width, u32 *actual_speed);

#endif /* NIC_H */
