#!/bin/bash

checkpach_mandate=1
sparse_mandate=1
coverity_mandate=1
checkpatch=tools/checkpatch/checkpatch.pl
cov_path=/tools/batonroot/coverity/2022.3.0/bin
cov_out_path=$(pwd)
compiler=gcc

if [ ! -e ${checkpatch} ]; then
	echo "***WARNING: Checkpatch file does not exist"
	checkpach_mandate=0
fi

function print_errs() {
	while read -r line; do
		echo $line
	done <<< "$1"
}

function run_checkpatch() {
	chmod +x ${checkpatch}
	make clean
	echo -e "\033[1mRunning checkpatch\033[0m"
	rm -f .tmp_cp_log
	mv drivers/net/ethernet/xilinx/mcdi_pcol.h drivers/net/ethernet/xilinx/mcdi_pcol.tmp #excluding mcdi_pcol.h in checkpatch
	mv drivers/net/ethernet/xilinx/efct_reg.h drivers/net/ethernet/xilinx/efct_reg.tmp #excluding efct_reg.h in checkpatch
	${checkpatch} --no-tree -f --show-types --ignore MACRO_ARG_REUSE drivers/net/ethernet/xilinx/*.c drivers/net/ethernet/xilinx/*.h drivers/net/ethernet/xilinx/util/*.c drivers/net/ethernet/xilinx/util/*.h > .tmp_cp_log
	mv drivers/net/ethernet/xilinx/mcdi_pcol.tmp drivers/net/ethernet/xilinx/mcdi_pcol.h
	mv drivers/net/ethernet/xilinx/efct_reg.tmp drivers/net/ethernet/xilinx/efct_reg.h
	cp_errs=$(cat .tmp_cp_log | grep "ERROR\|WARNING\|CHECK\|FILE")
	if [ "${cp_errs}" != "" ]; then
		echo "-------------------"
		echo "Checkpatch Errors:"
		echo "-------------------"
		print_errs "${cp_errs}"
		echo -e "\033[1m***CLEAR CHECKPATCH ERRORS. For full logs check .tmp_cp_log file***\033[0m"
	fi
}

function run_sparse() {
	echo -e "\033[1mRunning Sparse tool\033[0m"
	make clean
	make C=1 2>&1 | tee sparse_output
}

function run_coverity() {
	echo -e "\033[1mRunning Coverity\033[0m"
	export PATH=${cov_path}:$PATH
	make clean
	rm -rf ${cov_out_path}/coverity
	mkdir -p ${cov_out_path}/coverity/config
	cov-configure --template --config ${cov_out_path}/coverity/config/coverity_config.xml --compiler ${compiler}
	cov-build --config ${cov_out_path}/coverity/config/coverity_config.xml --dir ${cov_out_path}/coverity/config make
	if [ $? -ne 0 ]; then
		echo "cov-build failed"
		exit 1
	fi
	cov-analyze --all --allow-unmerged-emits --aggressiveness-level high --enable-parse-warnings -enable-default --enable-exceptions --enable-fnptr --config ${cov_out_path}/coverity/config/coverity_config.xml --dir ${cov_out_path}/coverity/config
	if [ $? -ne 0 ]; then
		echo "cov-analyze failed"
		exit 1
	fi
	cov-format-errors --config ${cov_out_path}/coverity/config/coverity_config.xml --html-output ${cov_out_path}/coverity/config/formatted-output --dir ${cov_out_path}/coverity/config
	echo -e "\033[1mCoverity logs are at ${cov_out_path}/coverity/config/formatted-output \033[0m"
}

function compile_driver() {
	echo -e "\033[1mCompiling the driver\033[0m"
	make clean
	make
}

if [ ${checkpach_mandate} -ne 0 ]; then
	run_checkpatch
fi

if [ ${sparse_mandate} -ne 0 ]; then
	run_sparse
fi

if [ ${coverity_mandate} -ne 0 ]; then
	run_coverity
fi

compile_driver
