#!/bin/bash

################################################################################
#
# Driver for Xilinx network controllers and boards
# Copyright 2021 Xilinx Inc.
#
# This program is free software; you can redistribute it and/or modify it
# under the terms of the GNU General Public License version 2 as published
# by the Free Software Foundation, incorporated herein by reference.
#
################################################################################

######################################################################
# This script runs as root inside the chroot setup in rpmbuild-master

me=$(basename "$0")
bin=$(cd "$(dirname "$0")" && /bin/pwd)

err  () { echo 1>&2 "$*";   }
log  () { err "$me: $*";     }
vlog () { $verbose && err "$me: $*"; }
fail () { log "$*"; exit 1; }
try  () { "$@" || fail "'$*' failed"; }
trace() { log "invoking: $*"; "$@"; }
vtrace() { vlog "invoking: $*"; "$@"; }


buildinstalllib() {
  d=$(mktemp -d /tmp/rpmbuild.XXXXXX)
  (
    try cp "$1" "$d"
    try cd "$d"
    cf="-O2 -shared -fpic -nostartfiles -ldl"
    nm=$(basename "${1%%.c}")
    if [ -d /usr/lib64 ]; then
      try rm -f "$nm"
      try trace make "$nm" CFLAGS="$cf"
      try trace install "$nm" "/usr/lib64/$nm.so"
      cf="$cf -m32"
    fi
    try rm -f "$nm"
    try trace make "$nm" CFLAGS="$cf"
    try trace install "$nm" "/usr/lib/$nm.so"
  ); rc=$?
  rm -rf "$d"
  return $rc
}

######################################################################

try [ $# = 7 ]
tag="$1"
kernels_file="$2"
dir="$3"
user="$4"
disttag="$5"
type="$6"
ptype_wanted="$7"

vlog "tag=$tag kernels_file=$kernels_file dir=$dir user=$user"\
  "type=$type"

try [ -d "$dir" ]
try [ -f "$kernels_file" ]

# Preloading libfakearch allows us to intercept uname(2) and make it look
# like we're running under a different architecture from the one we're
# really running under.  rpm-builder etc. set FAKEARCH_MACHINE to complete
# the magic.  If libfakearch is not present or old, we'd better build it.
c="$bin/libfakearch.c"
try [ -f "$c" ]
so="/usr/lib/libfakearch.so"
if [ ! -x "$so" ] || [ "$c" -nt "$so" ]; then
  buildinstalllib "$c"
fi

anybad=false

# Note that this splits the config file at ":"
# Format of config file <builder>:<arch>:<kernel:<service pack>
while IFS=: read ptype arch kernel sp; do
    distro_arch=$(basename "$kernels_file")
    distro=${distro_arch%_*}
    [ -n "$sp" ] || sp="base"

    if [ "$ptype_wanted" != "all" -a "$ptype_wanted" != "$ptype" ]; then
	continue
    fi
    if [ -z "${ptype%%\#*}" ]; then
	continue
    fi
    if [ ! -x "$bin/$ptype-builder" ]; then
	help
    fi

    # Run the builder (as a different user and with libfakearch)
    vtrace su - "$user" -c "LD_PRELOAD=libfakearch.so $bin/$ptype-builder $($verbose && echo -v) --disttag \"$disttag\" \"$dir\" $tag $type $distro $sp $arch:$kernel"
    rc=$?
    vlog "package builder exited rc=$rc"
    if [ $rc -ne 0 ]; then
	anybad=true
    fi

    # builder-post is called in rpmbuild-master as root in a similar
    # loop around all kernels

done < "$kernels_file"

! $anybad
